/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page;

import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.runtime.state.gemini.engine.dbms.Executor;
import org.apache.flink.runtime.state.gemini.engine.memstore.GSValue;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * PageStore.
 */
public interface PageStore<K, V> extends Executor {

	V get(K key);

	boolean contains(K key);

	/**
	 * Get all key-value pairs and store them in the given container if it does not exist.
	 * NOTE: expired keys will not be returned.
	 */
	void getAll(Map<K, GSValue<V>> container);

	PageIndex<K> getPageIndex();

	/**
	 * write data to Page, and register to mapping table, update index.
	 * if necessary, can do merge directly.
	 *
	 * @param pageIndexContext pageIndexContext is context information for this page.
	 * @param dataSet          data which will be written to page. if null, means only merge old page.
	 * @param version          current snapshot version.
	 */
	void addPage(PageIndexContext pageIndexContext, List<Tuple2<K, GSValue<V>>> dataSet, long version);

	/**
	 * compact Page dataPage. and only split once per each splitPage operator.
	 *
	 * @param pageIndexContext pageIndexContext is context information for this page.
	 * @param version          current snapshot version.
	 */
	void compactPage(PageIndexContext pageIndexContext, long version);

	/**
	 * merge two of logic data Page.
	 */
	void mergePage(PageIndexContext pageIndexContextFirst, PageIndexContext pageIndexContextSecond);

	/**
	 * split dataPage. and only split once per each splitPage operator.
	 *
	 * @param pageIndexContext pageIndexContext is context information for this page.
	 */
	void splitPage(PageIndexContext pageIndexContext);

	/**
	 * checkResource do check page size to decide whether to do expand index size.
	 */
	void checkResource();

	/**
	 * get all keys in pageStore, including deleted keys.
	 */
	void allKeysIncludeDeleted(Set<K> allKeysIncludeDelete);
}
