/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.core.memory.DataInputView;
import org.apache.flink.runtime.state.gemini.engine.GRegion;
import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiRuntimeException;
import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiShutDownException;
import org.apache.flink.runtime.state.gemini.engine.memstore.GSValue;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.BinaryKey;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.BinaryValue;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.ByteBufferDataInputView;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinaryHashMap;
import org.apache.flink.runtime.state.gemini.engine.rm.ReferenceCount.ReleaseType;
import org.apache.flink.util.Preconditions;

import org.apache.flink.shaded.netty4.io.netty.util.concurrent.EventExecutor;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinaryHashMap.EMPTY_G_BINARY_HASHMAP;

/**
 * PageStoreHashKListImpl.
 */
public class PageStoreHashKListImpl<K, E> extends AbstractHashPageStore<K, List<GSValue<E>>> implements PageStoreKList<K, E> {
	private static final Logger LOG = LoggerFactory.getLogger(PageStoreHashKListImpl.class);
	protected PageSerdeFlinkListImpl<K, E> pageSerdeFlinkListImpl;

	public PageStoreHashKListImpl(
		GRegion gRegion, EventExecutor eventExecutor) {
		super(gRegion, eventExecutor);
		this.pageSerdeFlinkListImpl = (PageSerdeFlinkListImpl<K, E>) gRegionContext.getPageSerdeFlink();
	}

	public PageStoreHashKListImpl(
		GRegion gRegion, PageIndex pageIndex, EventExecutor eventExecutor) {
		super(gRegion, pageIndex, eventExecutor);
		this.pageSerdeFlinkListImpl = (PageSerdeFlinkListImpl<K, E>) gRegionContext.getPageSerdeFlink();
	}

	@Override
	public boolean contains(K key) {
		//GeminiDB define empty collection as null.
		List<GSValue<E>> result = get(key);
		return get(key) != null && !result.isEmpty();
	}

	@Override
	long getRequestCount(List<Tuple2<K, GSValue<List<GSValue<E>>>>> dataSet) {
		return dataSet.stream().map((value) -> value.f1.getRequestCount()).reduce(0, (a, b) -> a + b);
	}

	@Override
	DataPage doCreateDataPage(
		long version, List<Tuple2<K, GSValue<List<GSValue<E>>>>> dataSet, int logicPageId) {
		GBinaryHashMap<K> gBinaryHashMap = GBinaryHashMap.of(DataPage.DataPageType.KList,
			dataSet,
			this.pageSerdeFlink.getKeySerde(),
			pageSerdeFlinkListImpl.getgListValueTypeSerializer(),
			version,
			logicPageId,
			gContext.getSupervisor().getAllocator(),
			1,
			gRegionContext.getGContext().getInPageGCompressAlgorithm());
		return gBinaryHashMap == EMPTY_G_BINARY_HASHMAP
			? null
			: new DataPageKListImpl<>(gBinaryHashMap,
				pageSerdeFlinkListImpl.getValueSerde(),
				pageSerdeFlinkListImpl.getgListValueTypeSerializer());
	}

	@Override
	public List<GSValue<E>> get(K key) {
		final PageIndexContext pageIndexContext = pageIndex.getPageIndexContext(key, false);
		final LogicChainedPage logicPageID = pageIndexContext.getPageID();

		if (isNullPage(logicPageID)) {
			return null;
		}

		List<GSValue<E>> finalResult = null;
		int curIndex = logicPageID.getCurrentPageChainIndex();
		List<BinaryValue> binaryValueReversedOrderList = new ArrayList<>();

		Map<Integer, DataPage> fetchedDataPageMap = new HashMap<>();

		while (curIndex >= 0 && gContext.isDBNormal()) {
			DataPage dataPage = getDataPageAutoLoadIfNeed(logicPageID, curIndex, fetchedDataPageMap);
			Preconditions.checkArgument(dataPage instanceof DataPageKList, "Interal BUG, error page");
			logicPageID.getPageAddress(curIndex).addRequestCount(1);
			gRegionContext.getPageStoreStats().addPageRequestCount(1);
			BinaryValue binaryValue = dataPage.getBinaryValue(key);
			dataPage.delReferenceCount(ReleaseType.Normal);
			if (binaryValue != null) {
				if (binaryValue.getgValueType() == GValueType.Delete) {
					//old value is useless
					break;
				} else {
					binaryValueReversedOrderList.add(binaryValue);
					if (binaryValue.getgValueType() == GValueType.PutList) {
						//old value is useless
						break;
					}
				}
			}
			curIndex--;
		}
		if (!gContext.isDBNormal()) {
			throw new GeminiShutDownException("DB is in abnormal status " + gContext.getDBStatus().name());
		}

		if (binaryValueReversedOrderList.size() == 0) {
			finalResult = null;
		} else {
			finalResult = doCompactValueToPOJO(binaryValueReversedOrderList);
		}

		tryLaunchCompactionByRead(pageIndexContext, logicPageID, fetchedDataPageMap);
		// NOTE: state will be filtered by the upper level
		return finalResult;
	}

	protected List<GSValue<E>> doCompactValueToPOJO(List<BinaryValue> binaryValueReversedOrderList) {
		try {
			List<GSValue<E>> result = new ArrayList<>();
			int index = binaryValueReversedOrderList.size() - 1;
			while (index >= 0) {
				BinaryValue binaryValue = binaryValueReversedOrderList.get(index);
				DataInputView byteBufferDataInputView = new ByteBufferDataInputView(binaryValue.getBb(),
					binaryValue.getValueOffset(),
					binaryValue.getValueLen());
				List<GSValue<E>> gsValueList = pageSerdeFlinkListImpl.getgListValueTypeSerializer().deserialize(
					byteBufferDataInputView);
				if (gsValueList == null) {
					continue;
				}
				result.addAll(gsValueList);
				index--;
			}
			return result;
		} catch (Exception e) {
			throw new GeminiRuntimeException("Exception: " + e.getMessage(), e);
		}
	}

	@Override
	public void getAll(Map<K, GSValue<List<GSValue<E>>>> container) {
		// TODO
	}

	@Override
	public DataPage doCompactPage(
		boolean isMajor, List<DataPage> canCompactPageListReversedOrder, long version, int logicPageId) {
		if (canCompactPageListReversedOrder == null || canCompactPageListReversedOrder.size() == 0) {
			throw new GeminiRuntimeException("Interal BUG");
		}
		return doCompactPageForStructureValue(isMajor, canCompactPageListReversedOrder, version, logicPageId);
	}

	@Override
	protected DataPage doBuildDataPageFromGBinaryMap(
		boolean isMajor,
		long version,
		int logicPageId,
		TypeSerializer<K> keySerde,
		Map<BinaryKey, BinaryValue> finalCompactedMap,
		long compactionCount) {
		GBinaryHashMap<K> gBinaryHashMap = GBinaryHashMap.ofBinaryList(DataPage.DataPageType.KList,
			isMajor,
			version,
			logicPageId,
			this.pageSerdeFlink.getKeySerde(),
			gContext.getSupervisor().getAllocator(),
			finalCompactedMap,
			compactionCount,
			gContext.getStateFilter(),
			gRegionContext);

		//TODO delReference finalCompactedMap.

		return gBinaryHashMap == EMPTY_G_BINARY_HASHMAP
			? null
			: new DataPageKListImpl<>(gBinaryHashMap,
				pageSerdeFlinkListImpl.getValueSerde(),
				pageSerdeFlinkListImpl.getgListValueTypeSerializer());
	}

	@Override
	BinaryValue doCompactValue(
		List<BinaryValue> binaryValueList, boolean isMajor, long version, int logicPageId) {
		return DataPageKListImpl.doCompactionKListValue(binaryValueList,
			pageSerdeFlink.getValueSerde(),
			isMajor,
			version,
			logicPageId,
			//value use DefaultAllocator
			gContext.getSupervisor().getDefaultAllocator());
	}

}
