/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.runtime.state.gemini.engine.GRegion;
import org.apache.flink.runtime.state.gemini.engine.memstore.GSValue;
import org.apache.flink.runtime.state.gemini.engine.page.DataPage.DataPageType;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.BinaryKey;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.BinaryValue;
import org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinaryHashMap;
import org.apache.flink.util.Preconditions;

import org.apache.flink.shaded.guava18.com.google.common.collect.Maps;
import org.apache.flink.shaded.netty4.io.netty.util.concurrent.EventExecutor;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import static org.apache.flink.runtime.state.gemini.engine.page.bmap.GBinaryHashMap.EMPTY_G_BINARY_HASHMAP;

/**
 * PageStoreHashKSortedMapImpl.
 */
public class PageStoreHashKSortedMapImpl<K, MK, MV> extends PageStoreHashKMapImpl<K, MK, MV> implements PageStoreKSortedMap<K, MK, MV> {
	private static final Logger LOG = LoggerFactory.getLogger(PageStoreHashKSortedMapImpl.class);

	public PageStoreHashKSortedMapImpl(
		GRegion gRegion, EventExecutor eventExecutor) {
		super(gRegion, eventExecutor);
	}

	public PageStoreHashKSortedMapImpl(
		GRegion gRegion, PageIndex pageIndex, EventExecutor eventExecutor) {
		super(gRegion, pageIndex, eventExecutor);
	}

	@Override
	DataPage doCreateDataPage(
		long version, List<Tuple2<K, GSValue<Map<MK, GSValue<MV>>>>> dataSet, int logicPageId) {
		GBinaryHashMap<K> gBinaryHashMap = GBinaryHashMap.of(DataPageType.KSortedMap,
			dataSet,
			this.pageSerdeFlink.getKeySerde(),
			this.pageSerdeFlink2Key.getMapValueTypeSerializer(),
			version,
			logicPageId,
			gContext.getSupervisor().getAllocator(),
			1,
			gRegionContext.getGContext().getInPageGCompressAlgorithm());
		return gBinaryHashMap == EMPTY_G_BINARY_HASHMAP
			? null
			: new DataPageKSortedMapImpl<>(gBinaryHashMap,
				this.pageSerdeFlink2Key.getKey2Serde(),
				this.pageSerdeFlink2Key.getValueSerde(),
				this.pageSerdeFlink2Key.getMapValueTypeSerializer(),
				this.pageSerdeFlink2Key.getMapComparator());
	}

	@Override
	public TreeMap<MK, GSValue<MV>> get(K key) {
		Map<MK, GSValue<MV>> result = super.get(key);
		if (result != null) {
			Preconditions.checkArgument(result instanceof TreeMap, "Interal Bug");
			// NOTE: state will be filtered by the upper level
			return (TreeMap<MK, GSValue<MV>>) result;
		} else {
			return null;
		}
	}

	@Override
	TreeMap<MK, GSValue<MV>> doCompactValueToPOJO(List<BinaryValue> binaryValueReversedOrderList) {
		Map<BinaryKey, BinaryValue> binaryValueMap = DataPageKMapImpl.doCompactValueToBinaryMap(
			binaryValueReversedOrderList,
			pageSerdeFlink2Key.getKey2Serde());

		TreeMap<MK, GSValue<MV>> result = new TreeMap<>(this.pageSerdeFlink2Key.getMapComparator().getJDKCompactor());

		for (Map.Entry<BinaryKey, BinaryValue> entry : binaryValueMap.entrySet()) {
			//For POJO, it results to client, so it don't need to include the Deleted value.
			if (entry.getValue() == null || entry.getValue().getgValueType() == GValueType.Delete) {
				continue;
			}
			result.put(getMKeyFromBinary(entry.getKey()), getMValueFromBinary(entry.getValue()));
		}
		return result;
	}

	@Override
	protected DataPage doBuildDataPageFromGBinaryMap(
		boolean isMajor,
		long version,
		int logicPageId,
		TypeSerializer<K> keySerde,
		Map<BinaryKey, BinaryValue> finalCompactedMap,
		long compactionCount) {
		GBinaryHashMap<K> gBinaryHashMap = GBinaryHashMap.ofBinaryList(DataPageType.KSortedMap,
			isMajor,
			version,
			logicPageId,
			this.pageSerdeFlink.getKeySerde(),
			gContext.getSupervisor().getAllocator(),
			finalCompactedMap,
			compactionCount,
			gContext.getStateFilter(),
			gRegionContext);

		//TODO delReference finalCompactedMap.
		return gBinaryHashMap == EMPTY_G_BINARY_HASHMAP
			? null
			: new DataPageKSortedMapImpl<>(gBinaryHashMap,
				this.pageSerdeFlink2Key.getKey2Serde(),
				this.pageSerdeFlink2Key.getValueSerde(),
				this.pageSerdeFlink2Key.getMapValueTypeSerializer(),
				this.pageSerdeFlink2Key.getMapComparator());
	}

	@Override
	BinaryValue doCompactValue(
		List<BinaryValue> binaryValueList, boolean isMajor, long version, int logicPageId) {
		return DataPageKSortedMapImpl.doCompactionSortedMapValue(binaryValueList,
			pageSerdeFlink2Key.getKey2Serde(),
			pageSerdeFlink2Key.getMapComparator(),
			isMajor,
			version,
			logicPageId,
			//value use defaultAllocator
			gContext.getSupervisor().getDefaultAllocator(),
			gContext.getStateFilter(),
			gRegionContext);
	}

	@Override
	public Map.Entry<MK, MV> firstEntry(K key) {
		//TODO now only support get the POJO treeMap. to support sort-merge In future.
		TreeMap<MK, GSValue<MV>> sortedMap = get(key);
		if (sortedMap == null) {
			return null;
		}
		//No deleted of type in there.
		Map.Entry<MK, GSValue<MV>> firstEntry = sortedMap.firstEntry();

		return new Map.Entry<MK, MV>() {
			@Override
			public MK getKey() {
				return firstEntry.getKey();
			}

			@Override
			public MV getValue() {
				return firstEntry.getValue().getValue();
			}

			@Override
			public MV setValue(MV value) {
				return null;
			}
		};
	}

	@Override
	public Map.Entry<MK, MV> lastEntry(K key) {
		//TODO now only support get the POJO treeMap. to support sort-merge In future.
		TreeMap<MK, GSValue<MV>> sortedMap = get(key);
		if (sortedMap == null) {
			return null;
		}
		//No deleted of type in there.
		Map.Entry<MK, GSValue<MV>> lastEntry = sortedMap.lastEntry();
		return new Map.Entry<MK, MV>() {
			@Override
			public MK getKey() {
				return lastEntry.getKey();
			}

			@Override
			public MV getValue() {
				return lastEntry.getValue().getValue();
			}

			@Override
			public MV setValue(MV value) {
				return null;
			}
		};
	}

	@Override
	public Iterator<Map.Entry<MK, MV>> headIterator(K key, MK endMapKey) {
		//TODO now only support get the POJO treeMap. to support sort-merge In future.
		TreeMap<MK, GSValue<MV>> sortedMap = get(key);
		if (sortedMap == null) {
			Collections.emptyIterator();
		}
		//No deleted of type in there.
		SortedMap<MK, GSValue<MV>> submap = sortedMap.headMap(endMapKey);
		if (submap == null || submap.size() == 0) {
			Collections.emptyIterator();
		}
		return Maps.transformEntries(submap, (mk, mv) -> mv.getValue()).entrySet().iterator();
	}

	@Override
	public Iterator<Map.Entry<MK, MV>> tailIterator(K key, MK startMapKey) {
		//TODO now only support get the POJO treeMap. to support sort-merge In future.
		TreeMap<MK, GSValue<MV>> sortedMap = get(key);
		if (sortedMap == null) {
			Collections.emptyIterator();
		}
		//No deleted of type in there.
		SortedMap<MK, GSValue<MV>> submap = sortedMap.tailMap(startMapKey);
		if (submap == null || submap.size() == 0) {
			Collections.emptyIterator();
		}
		return Maps.transformEntries(submap, (mk, mv) -> mv.getValue()).entrySet().iterator();
	}

	@Override
	public Iterator<Map.Entry<MK, MV>> subIterator(K key, MK startMapKey, MK endMapKey) {
		//TODO now only support get the POJO treeMap. to support sort-merge In future.
		TreeMap<MK, GSValue<MV>> sortedMap = get(key);
		if (sortedMap == null) {
			Collections.emptyIterator();
		}
		//No deleted of type in there.
		SortedMap<MK, GSValue<MV>> submap = sortedMap.subMap(startMapKey, endMapKey);
		if (submap == null || submap.size() == 0) {
			Collections.emptyIterator();
		}
		return Maps.transformEntries(submap, (mk, mv) -> mv.getValue()).entrySet().iterator();
	}
}
