/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page;

import org.apache.flink.runtime.state.gemini.engine.GRegion;
import org.apache.flink.runtime.state.gemini.engine.vm.CacheManager;

import org.apache.flink.shaded.guava18.com.google.common.base.MoreObjects;
import org.apache.flink.shaded.netty4.io.netty.util.concurrent.EventExecutor;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

/**
 * stats about page, including the number of page, and it's average size, etc.
 */
public class PageStoreStats {
	private final CacheManager cacheManager;
	private int indexCapacity = 0;
	//every time transforming POJO data page to the Heap data page, this rate will be adjusted.
	private volatile float pageSizeRate;
	private final AtomicInteger logicPageCount = new AtomicInteger(0);
	private final AtomicLong pageUsedMemory = new AtomicLong(0);
	private final AtomicLong logicPageSize = new AtomicLong(0);
	private final AtomicLong logicPageChainCapacity = new AtomicLong(0);
	private final AtomicLong logicPageChainLen = new AtomicLong(0);
	private final AtomicLong pageRequestCount = new AtomicLong(0);
	private volatile float evictedRate = 0.0f;
	private final AtomicLong addPages = new AtomicLong(0);
	private final AtomicLong minorCompactions = new AtomicLong(0);
	private final AtomicLong minorCompactedPages = new AtomicLong(0);
	private final AtomicLong majorCompactions = new AtomicLong(0);
	private final AtomicLong majorCompactedPages = new AtomicLong(0);
	private final AtomicInteger runningMinorCompaction = new AtomicInteger(0);
	private final AtomicInteger runningMajorCompaction = new AtomicInteger(0);
	private final AtomicInteger runningMinorCompactionByRead = new AtomicInteger(0);
	private final String name;
	private EventExecutor regionEventExecutor;

	public PageStoreStats(String name, CacheManager cacheManager) {
		this.name = name;
		this.cacheManager = cacheManager;
	}

	public int getIndexCapacity() {
		return indexCapacity;
	}

	public float getPageSizeRate() {
		return pageSizeRate;
	}

	public int getLogicPageCount() {
		return logicPageCount.get();
	}

	public long getLogicPageSize() {
		return logicPageSize.get();
	}

	public void setIndexCapacity(int indexCapacity) {
		cacheManager.addIndexCapacity(-this.indexCapacity);
		this.indexCapacity = indexCapacity;
		cacheManager.addIndexCapacity(this.indexCapacity);
	}

	public void setPageSizeRate(float pageSizeRate) {
		this.pageSizeRate = pageSizeRate;
	}

	public void addLogicPageCount(int logicPageCount) {
		this.logicPageCount.addAndGet(logicPageCount);
		cacheManager.getCacheStats().addTotalLogicPageCount(logicPageCount);
	}

	public void addLogicPageSize(int logicPageSize) {
		this.logicPageSize.addAndGet(logicPageSize);
		this.cacheManager.getCacheStats().addTotalLogicPageSize(logicPageSize);
	}

	public void addPageUsedMemory(GRegion gRegion, int logicPageSize) {
		addPageUsedMemory(gRegion, logicPageSize, true);
	}

	public void addPageUsedMemory(GRegion gRegion, int logicPageSize, boolean needEvict) {
		this.pageUsedMemory.addAndGet(logicPageSize);
		this.cacheManager.getCacheStats().addPageUsedMemory(logicPageSize);

		this.cacheManager.getEvictPolicy().addPageUsedMemory(gRegion, this.regionEventExecutor, logicPageSize, needEvict);
	}

	public void addLogicPageChainCapacity(int logicPageChainLen) {
		this.logicPageChainCapacity.addAndGet(logicPageChainLen);
	}

	public void addLogicPageChainLen(int logicPageChainLen) {
		this.logicPageChainLen.addAndGet(logicPageChainLen);
		this.cacheManager.getCacheStats().addTotalLogicPageChainLen(logicPageChainLen);
	}

	public void addPageRequestCount(long pageRequest) {
		this.pageRequestCount.addAndGet(pageRequest);
	}

	public void addPage() {
		this.addPages.addAndGet(1);
	}

	public void addMinorCompactedPages(int n) {
		this.minorCompactions.addAndGet(1);
		this.minorCompactedPages.addAndGet(n);
		this.cacheManager.getCacheStats().addMinorCompactedPages(n);
	}

	public void addMajorCompactedPages(int n) {
		this.majorCompactions.addAndGet(1);
		this.majorCompactedPages.addAndGet(n);
		this.cacheManager.getCacheStats().addMajorCompactedPages(n);
	}

	public void addRuningMajorCompactedPages(int i) {
		this.runningMajorCompaction.addAndGet(i);
		this.cacheManager.getCacheStats().addTotalRuningMajorCompactedPages(i);
	}

	public void addRuningMinorCompactedPages(int i) {
		this.runningMinorCompaction.addAndGet(i);
		this.cacheManager.getCacheStats().addTotalRuningMinorCompactedPages(i);
	}

	public void addRuningMinorCompactionByRead(int i) {
		this.runningMinorCompactionByRead.addAndGet(i);
		this.cacheManager.getCacheStats().addRuningMinorCompactionByRead(i);
	}

	@Override
	public String toString() {
		MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper(this).
			add("name", name).
			add("indexCapacity", indexCapacity).
			add("pageSizeRate", pageSizeRate).
			add("logicPageCount", logicPageCount.get()).
			add("pageUsedMemory", pageUsedMemory.get()).
			add("logicPageSize", logicPageSize.get()).
			add("logicPageChainLen", logicPageChainLen.get()).
			add("logicPageChainCapacity", logicPageChainCapacity.get()).
			add("pageRequestCount", pageRequestCount.get()).
			add("evictedRate", evictedRate).
			add("addPages", addPages.get()).
			add("minorCompactions", minorCompactions.get()).
			add("minorCompactedPages", minorCompactedPages.get()).
			add("majorCompactions", majorCompactions.get()).
			add("majorCompactedPages", majorCompactedPages.get()).
			add("runningMajorCompaction", runningMajorCompaction.get()).
			add("runningMinorCompaction", runningMinorCompaction.get()).
			add("runningMinorCompactionByRead", runningMinorCompactionByRead.get());

		if (minorCompactions.get() > 0) {
			helper.add("avgSyncCompactedPages", ((double) minorCompactedPages.get()) / minorCompactions.get());
		}

		return helper.toString();
	}

	public void setRegionExecutor(EventExecutor regionEventExecutor) {
		this.regionEventExecutor = regionEventExecutor;
	}
}
