/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page.bmap;

import java.nio.ByteBuffer;

/**
 * BinaryKey for BinaryHashMap.
 */
public final class BinaryKey {
	private final int keyOffset;
	private final int keyLen;
	private final int keyhashCode;
	// BinaryKey is internal data structure, so no need GByteBuffer to release reference count.
	private final ByteBuffer bb;

	public BinaryKey(ByteBuffer data, int keyOffset, int keyLen, int keyhashCode) {
		this.keyOffset = keyOffset;
		this.keyLen = keyLen;
		this.bb = data;
		this.keyhashCode = keyhashCode;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (o == null || getClass() != o.getClass()) {
			return false;
		}
		BinaryKey binaryKey = (BinaryKey) o;
		return keyhashCode == binaryKey.keyhashCode && keyLen == binaryKey.keyLen && ByteBufferUtils.compareTo(bb,
			keyOffset,
			keyLen,
			binaryKey.bb,
			binaryKey.keyOffset,
			binaryKey.keyLen) == 0;
	}

	@Override
	public int hashCode() {
		return this.keyhashCode;
	}

	public int getKeyOffset() {
		return keyOffset;
	}

	public int getKeyLen() {
		return keyLen;
	}

	public int getKeyhashCode() {
		return keyhashCode;
	}

	public ByteBuffer getBb() {
		return bb;
	}
}
