/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page.bmap;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.core.memory.DataInputView;

import java.util.Comparator;

/**
 * GComparatorBytesImpl for Gemini, use bytes comparator.
 */
public class GComparatorBytesImpl<K> implements GComparator<K> {
	private final Comparator<K> jdkComparator;
	private final TypeSerializer<K> typeSerializer;

	public GComparatorBytesImpl(Comparator<K> jdkComparator, TypeSerializer<K> typeSerializer) {
		this.jdkComparator = jdkComparator;
		this.typeSerializer = typeSerializer;
	}

	@Override
	public int compare(BinaryKey left, BinaryKey right) {
		return ByteBufferUtils.compareTo(left.getBb(),
			left.getKeyOffset(),
			left.getKeyLen(),
			right.getBb(),
			right.getKeyOffset(),
			right.getKeyLen());
	}

	@Override
	public int compare(K left, K right) {
		return this.jdkComparator.compare(left, right);
	}

	@Override
	public int compare(K left, BinaryKey right) {
		try {
			DataInputView rightDataInputView = new ByteBufferDataInputView(right.getBb(),
				right.getKeyOffset(),
				right.getKeyLen());

			K rigtK = typeSerializer.deserialize(rightDataInputView);
			return this.jdkComparator.compare(left, rigtK);
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	@Override
	public Comparator<K> getJDKCompactor() {
		return this.jdkComparator;
	}

	@Override
	public Comparator<BinaryKey> getJDKBinaryCompactor() {
		return GComparatorBytesImpl.this::compare;
	}
}
