/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page.bmap;

import java.nio.ByteBuffer;

/**
 * GHashHeader.
 */
public interface GHashHeader {

	/**
	 * init index data.
	 */
	void initIndex(ByteBuffer headerAndIndex, int indexLen);

	/**
	 * get index value for a slot.
	 */
	long getIndexBySlot(ByteBuffer headerAndIndex, int indexSlot);

	/**
	 * write a index value for a slot.
	 */
	void writeIndexBySlot(ByteBuffer headerAndIndexBB, int indexSlot, long newIndexValue);

	/**
	 * write hash code to a certain slot(keyCursor).
	 */
	void writeHashCode(ByteBuffer headerAndIndexBB, int indexLen, int keyCursor, int hashCode);

	/**
	 * write TTL to a certain slot(keyCursor).
	 */
	void writeSeqIDBytSlot(
		ByteBuffer headerAndIndexBB, int indexLen, int totalKeys, long seqID, int keyCursor);

	/**
	 * get hash code for a certain slot(keyCursor).
	 */
	int getHashCode(ByteBuffer headerAndIndexBB, int indexLen, int keyCursor);

	/**
	 * get TTL for a certain slot(keyCursor).
	 */
	long getSeqIDBytSlot(
		ByteBuffer headerAndIndexBB, int indexLen, int totalKeys, int keyCursor);

	/**
	 * the index and head total length.
	 */
	int getHeaderAndIndexLen(int indexLen, int totalKeys);

	/**
	 * hash index depends on which index use, such as LongIndexPage(4 bytes count, 4 bytes indicator), IntegerIndex(2 bytes count, 2 bytes indicator).
	 *
	 * @param indexValue index value for a item.
	 * @return count represented by this index.
	 */
	int getCountByIndexValue(long indexValue);

	/**
	 * hash index depends on which index use, such as LongIndexPage(4 bytes count, 4 bytes indicator), IntegerIndex(2 bytes count, 2 bytes indicator).
	 *
	 * @param indexValue index value for a item.
	 * @return slot indicator represented by this index.
	 */
	int getSlotByIndexValue(long indexValue);

	/**
	 * initial index value.
	 *
	 * @return initial index value.
	 */
	long getInitialIndexValue();

	/**
	 * new index value will add count, and return new index value.
	 *
	 * @param oldIndexValue old index value.
	 * @param slot          slot indicator.
	 * @return new index value.
	 */
	long getNewIndexValue(long oldIndexValue, int slot);
}
