/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page.bmap;

import java.nio.ByteBuffer;

/**
 * GBinarySortedMap has both a hash index and sorted index. to offer faster lookup than common sortedMap.
 * GSortedHeader's different from GHashHeader is only GSortedHeader has a SORTED INDEX.
 * Header  64 bytes
 * hash index depends on which index use, such as LongIndexPage(4 bytes count, 4 bytes indicator), IntegerIndex(2 bytes count, 2 bytes indicator).
 * hash code  4 bytes * count
 * TTL  8 bytes * count
 * SORTED INDEX (4 bytes * count)
 * 4byte keyIndicator   //TODO if key is fixed-len, it will be omitted.
 * key set
 * 4byte valueIndicator  //TODO if value is fixed-len, it will be omitted.
 * value set
 */
public enum GSortedHeaderImpl implements GSortedHeader, GHashHeader {
	WithLongIndexPage(GHashHeaderImpl.LongIndexPage),
	WithIntegerIndexPage(GHashHeaderImpl.IntegerIndexPage);

	private final GHashHeaderImpl gHashHeader;

	GSortedHeaderImpl(GHashHeaderImpl gHashHeader) {
		this.gHashHeader = gHashHeader;
	}

	public GHashHeaderImpl getgHashHeader() {
		return gHashHeader;
	}

	public static GSortedHeaderImpl getPageHelper(int indexLen) {
		if (indexLen <= 65536) {
			return WithIntegerIndexPage;
		} else {
			return WithLongIndexPage;
		}
	}

	@Override
	public void initIndex(ByteBuffer headerAndIndex, int indexLen) {
		this.gHashHeader.initIndex(headerAndIndex, indexLen);
	}

	@Override
	public long getIndexBySlot(ByteBuffer headerAndIndex, int indexSlot) {
		return this.gHashHeader.getIndexBySlot(headerAndIndex, indexSlot);
	}

	@Override
	public void writeIndexBySlot(ByteBuffer headerAndIndexBB, int indexSlot, long newIndexValue) {
		this.gHashHeader.writeIndexBySlot(headerAndIndexBB, indexSlot, newIndexValue);
	}

	@Override
	public void writeHashCode(ByteBuffer headerAndIndexBB, int indexLen, int keyCursor, int hashCode) {
		this.gHashHeader.writeHashCode(headerAndIndexBB, indexLen, keyCursor, hashCode);
	}

	@Override
	public void writeSeqIDBytSlot(ByteBuffer headerAndIndexBB, int indexLen, int totalKeys, long seqID, int keyCursor) {
		this.gHashHeader.writeSeqIDBytSlot(headerAndIndexBB, indexLen, totalKeys, seqID, keyCursor);
	}

	@Override
	public int getHashCode(ByteBuffer headerAndIndexBB, int indexLen, int keyCursor) {
		return this.gHashHeader.getHashCode(headerAndIndexBB, indexLen, keyCursor);
	}

	@Override
	public long getSeqIDBytSlot(ByteBuffer headerAndIndexBB, int indexLen, int totalKeys, int keyCursor) {
		return this.gHashHeader.getSeqIDBytSlot(headerAndIndexBB, indexLen, totalKeys, keyCursor);
	}

	@Override
	public int getCountByIndexValue(long indexValue) {
		return this.gHashHeader.getCountByIndexValue(indexValue);
	}

	@Override
	public int getSlotByIndexValue(long indexValue) {
		return this.gHashHeader.getSlotByIndexValue(indexValue);
	}

	@Override
	public long getInitialIndexValue() {
		return this.gHashHeader.getInitialIndexValue();
	}

	@Override
	public long getNewIndexValue(long oldIndexValue, int slot) {
		return this.gHashHeader.getNewIndexValue(oldIndexValue, slot);
	}

	@Override
	public int getHeaderAndIndexLen(int indexLen, int totalKeys) {
		return this.gHashHeader.getHeaderAndIndexLen(indexLen, totalKeys) + totalKeys * Integer.BYTES;
	}

	//SORTED INDEX

	@Override
	public int getSortedIndexBaseOffset(int indexLen, int totalKeys) {
		return this.gHashHeader.getHeaderAndIndexLen(indexLen, totalKeys);
	}

	@Override
	public void writeSortedIndexBySlot(ByteBuffer headerAndIndexBB, int baseOffset, int indexSlot, long newIndexValue) {
		ByteBufferUtils.putInt(headerAndIndexBB, baseOffset + indexSlot * Integer.BYTES, (int) newIndexValue);
	}

	@Override
	public int getSortedIndexBySlot(ByteBuffer headerAndIndexBB, int baseOffset, int indexSlot) {
		return ByteBufferUtils.toInt(headerAndIndexBB, baseOffset + indexSlot * Integer.BYTES);
	}
}
