/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page.compress;

import org.apache.flink.api.common.io.blockcompression.AbstractBlockCompressor;
import org.apache.flink.api.common.io.blockcompression.AbstractBlockDecompressor;
import org.apache.flink.api.common.io.blockcompression.Lz4BlockCompressionFactory;
import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiRuntimeException;

import java.nio.ByteBuffer;

import static java.nio.ByteOrder.LITTLE_ENDIAN;

/**
 * CompressorCodec.
 * thread not safe.
 */
public class LZ4CompressorCodec implements CompressorCodec {

	private final AbstractBlockCompressor compressor;
	private final AbstractBlockDecompressor decompressor;
	private volatile byte[] reusedByteArray;
	private volatile ByteBuffer reusedOffheapBB = ByteBuffer.allocateDirect(16384);

	public LZ4CompressorCodec() {
		Lz4BlockCompressionFactory factory = new Lz4BlockCompressionFactory();
		compressor = factory.getCompressor();
		decompressor = factory.getDecompressor();
		reusedOffheapBB.order(LITTLE_ENDIAN);
	}

	@Override
	public byte[] getCompressedBytePool(int srcSize) {
		int size = compressor.getMaxCompressedSize(srcSize);
		if (reusedByteArray == null || reusedByteArray.length < size) {
			reusedByteArray = new byte[size];
		}
		return reusedByteArray;
	}

	@Override
	public ByteBuffer getReuseByteBuffer(int srcSize) {
		if (reusedOffheapBB == null || reusedOffheapBB.capacity() < srcSize) {
			reusedOffheapBB = ByteBuffer.allocateDirect(srcSize);
			reusedOffheapBB.order(LITTLE_ENDIAN);
		}
		reusedOffheapBB.clear().limit(srcSize);
		return reusedOffheapBB;
	}

	@Override
	public int getMaxCompressedSize(int srcSize) {
		return compressor.getMaxCompressedSize(srcSize);
	}

	@Override
	public int compress(ByteBuffer src, ByteBuffer dst) throws GeminiRuntimeException {
		return compressor.compress(src, dst);
	}

	@Override
	public int decompress(ByteBuffer src, ByteBuffer dst) throws GeminiRuntimeException {
		return decompressor.decompress(src, dst);
	}

	@Override
	public int compress(byte[] src, byte[] dst) throws GeminiRuntimeException {
		return compressor.compress(src, dst);
	}

	@Override
	public int decompress(byte[] src, byte[] dst) throws GeminiRuntimeException {
		return decompressor.decompress(src, dst);
	}

}
