/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.page.compress;

import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiRuntimeException;

import org.xerial.snappy.Snappy;

import java.io.IOException;
import java.nio.ByteBuffer;

import static java.nio.ByteOrder.LITTLE_ENDIAN;

/**
 * CompressorCodec.
 * thread not safe.
 */
public class SnappyCompressorCodec implements CompressorCodec {

	private volatile byte[] reusedByteArray;
	private volatile ByteBuffer reusedOffheapBB = ByteBuffer.allocateDirect(16384);

	public SnappyCompressorCodec() {
		reusedOffheapBB.order(LITTLE_ENDIAN);
	}

	@Override
	public byte[] getCompressedBytePool(int srcSize) {
		if (reusedByteArray == null || reusedByteArray.length < srcSize) {
			reusedByteArray = new byte[srcSize];
		}
		return reusedByteArray;
	}

	@Override
	public ByteBuffer getReuseByteBuffer(int srcSize) {
		if (reusedOffheapBB == null || reusedOffheapBB.capacity() < srcSize) {
			reusedOffheapBB = ByteBuffer.allocateDirect(srcSize);
			reusedOffheapBB.order(LITTLE_ENDIAN);
		}
		reusedOffheapBB.clear().limit(srcSize);
		return reusedOffheapBB;
	}

	@Override
	public int getMaxCompressedSize(int srcSize) {
		return srcSize;
	}

	@Override
	public int compress(ByteBuffer src, ByteBuffer dst) throws GeminiRuntimeException {
		try {
			return Snappy.compress(src, dst);
		} catch (IOException e) {
			throw new GeminiRuntimeException(e);
		}
	}

	@Override
	public int decompress(ByteBuffer src, ByteBuffer dst) throws GeminiRuntimeException {
		try {
			return Snappy.uncompress(src, dst);
		} catch (IOException e) {
			throw new GeminiRuntimeException(e);
		}
	}

	@Override
	public int compress(byte[] src, byte[] dst) throws GeminiRuntimeException {
		try {
			return Snappy.compress(src, 0, src.length, dst, 0);
		} catch (IOException e) {
			throw new GeminiRuntimeException(e);
		}
	}

	@Override
	public int decompress(byte[] src, byte[] dst) throws GeminiRuntimeException {
		try {
			return Snappy.uncompress(src, 0, src.length, dst, 0);
		} catch (IOException e) {
			throw new GeminiRuntimeException(e);
		}
	}

}
