/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.rm;

import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiRuntimeException;

import java.util.concurrent.atomic.AtomicInteger;

/**
 * GByteBuffer.
 */
public abstract class AbstractGByteBufferRefence implements GByteBuffer {
	private final AtomicInteger referenceCount = new AtomicInteger(0);
	private volatile long seqID;
	protected final Allocator allocator;

	public AbstractGByteBufferRefence(Allocator allocator) {
		this.allocator = allocator;
	}

	@Override
	public void contain() {
		referenceCount.incrementAndGet();
	}

	@Override
	public void release(ReleaseType releaseType) throws GeminiRuntimeException {
		//Notice: not thread safe. DiscardOrEvictPageReleaseManager will handle this case.
		//when release to 0, we will do the logic described by DiscardOrEvictPageReleaseManager. and it's processed by allocator.
		int count = referenceCount.decrementAndGet();
		//allocator is null, means not need to release space.
		if (count == 0) {
			if (allocator != null) {
				allocator.release(releaseType, this);
			} else {
				doFree();
			}
		}
	}

	@Override
	public int getCnt() {
		return referenceCount.get();
	}

	@Override
	public void setSeqID(long seqID) {
		this.seqID = seqID;
	}

	@Override
	public long getSeqID() {
		return this.seqID;
	}
}
