/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.rm;

import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiRuntimeException;

import org.apache.flink.shaded.netty4.io.netty.buffer.ByteBuf;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.ByteBuffer;

/**
 * GByteBuffer.
 */
public class GPooledNettyByteBuffer extends AbstractGByteBufferRefence {
	private static final Logger LOG = LoggerFactory.getLogger(GPooledNettyByteBuffer.class);
	private final ByteBuffer byteBuffer;
	private volatile ByteBuf byteBuf;
	private final int len;

	public GPooledNettyByteBuffer(ByteBuf byteBuf, int len, Allocator allocator) {
		super(allocator);
		this.byteBuf = byteBuf;
		byteBuffer = byteBuf.nioBuffer(0, len);
		this.len = len;
		contain();
	}

	@Override
	public int capacity() {
		return this.len;
	}

	@Override
	public ByteBuffer getByteBuffer() {
		return this.byteBuffer;
	}

	@VisibleForTesting
	public ByteBuf getByteBuf() {
		return byteBuf;
	}

	@Override
	public void doFree() throws GeminiRuntimeException {
		if (byteBuf == null) {
			//It can't happen.
			return;
		}
		synchronized (this) {
			if (byteBuf != null) {
				if (!byteBuf.release()) {
					LOG.error("FATAL BUG!!! LEAK! pls contact to dev. byteBuf ref =" + byteBuf.refCnt());
				}
				byteBuf = null;
				this.allocator.statSize(-len);
			}
		}
	}
}
