/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.rm;

import org.apache.flink.runtime.state.gemini.engine.GConfiguration;
import org.apache.flink.runtime.state.gemini.engine.dbms.DiscardOrEvictPageReleaseManager;
import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiException;
import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiRuntimeException;
import org.apache.flink.runtime.state.gemini.engine.rm.ReferenceCount.ReleaseType;

import org.apache.flink.shaded.guava18.com.google.common.base.MoreObjects;
import org.apache.flink.shaded.netty4.io.netty.buffer.ByteBuf;
import org.apache.flink.shaded.netty4.io.netty.buffer.PooledByteBufAllocator;

import java.util.concurrent.atomic.AtomicLong;

/**
 * PoolAllocatorNettyImpl. Now we use Netty PooledByteBufAllocator, but In future, we will use our Allocator to increase
 * memory utilization and performance.
 */
public class PoolAllocatorNettyImpl implements Allocator {
	private final PooledByteBufAllocator pooledByteBufAllocator;
	private final AtomicLong currentSize = new AtomicLong(0);
	private final DiscardOrEvictPageReleaseManager discardOrEvictPageReleaseManager;

	public PoolAllocatorNettyImpl(
		GConfiguration gConfiguration, DiscardOrEvictPageReleaseManager discardOrEvictPageReleaseManager) {
		//other parameter need to config.
		int pageSize = 8192;
		int maxOrder = 11;
		int normalCacheSize = 16;
		int nDirectArena = Math.min(gConfiguration.getAllocatorDirectArena(),
			(int) (gConfiguration.getTotalOffheapMemSize() / (pageSize << maxOrder) / 3 / 2));
		pooledByteBufAllocator = new PooledByteBufAllocator(true,
			0,
			nDirectArena,
			pageSize,
			maxOrder,
			0,
			0,
			normalCacheSize);
//		Field writeBufferSizeField = getDeclaredField(pooledByteBufAllocator, "writeBufferSize");
		this.discardOrEvictPageReleaseManager = discardOrEvictPageReleaseManager;

	}

	@Override
	public GByteBuffer allocate(int len) throws GeminiException {
		ByteBuf byteBuf = pooledByteBufAllocator.buffer(len);
		currentSize.addAndGet(len);
		return new GPooledNettyByteBuffer(byteBuf, len, this);
	}

	@Override
	public void close() {
		//do nothing
	}

	@Override
	public void release(ReleaseType releaseType, GByteBuffer gByteBuffer) {
		switch (releaseType) {
			case Discard:
			case Evict:
				this.discardOrEvictPageReleaseManager.addWaitSafeQueue(releaseType, gByteBuffer);
				break;
			case Normal:
				//Normal don't need to handle. because it will not suffer the multithread situation.
				gByteBuffer.doFree();
				break;
			default:
				throw new GeminiRuntimeException("release invalid type:" + releaseType);
		}

	}

	@Override
	public void statSize(int len) {
		currentSize.addAndGet(len);
	}

	@Override
	public String toString() {
		return MoreObjects.toStringHelper(this).
			add("pooledByteBufAllocator", pooledByteBufAllocator).
			add("currentSize", currentSize).toString();
//			add("metrics", pooledByteBufAllocator.metric()).toString();
		//current netty don't have metrics. new version will have. add("metric", pooledByteBufAllocator.)
	}
}

