/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.snapshot;

import org.apache.flink.runtime.state.SnapshotDirectory;

import javax.annotation.Nullable;

import java.util.Objects;

/**
 * Encapsulates all meta data used by GeminiDB to start a snapshot.
 * This is created in backend and delivered to DB.
 */
public class BackendSnapshotMeta {

	/**
	 * The ID of the checkpoint.
	 */
	private final long checkpointId;

	/**
	 * The timestamp of the checkpoint.
	 */
	private final long timestamp;

	/**
	 * Directory for local snapshot.
	 */
	@Nullable
	private final SnapshotDirectory localSnapshotDir;

	public BackendSnapshotMeta(
		long checkpointId,
		long timestamp) {
		this(checkpointId, timestamp, null);
	}

	public BackendSnapshotMeta(
		long checkpointId,
		long timestamp,
		SnapshotDirectory localSnapshotDir) {
		this.checkpointId = checkpointId;
		this.timestamp = timestamp;
		this.localSnapshotDir = localSnapshotDir;
	}

	public long getCheckpointId() {
		return checkpointId;
	}

	public long getTimestamp() {
		return timestamp;
	}

	public boolean isLocalSnapshotEnabled() {
		return getLocalSnapshotDir() != null;
	}

	public SnapshotDirectory getLocalSnapshotDir() {
		return localSnapshotDir;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (o == null || getClass() != o.getClass()) {
			return false;
		}

		BackendSnapshotMeta that = (BackendSnapshotMeta) o;

		return (checkpointId == that.checkpointId)
			&& (timestamp == that.timestamp)
			&& Objects.equals(localSnapshotDir, that.localSnapshotDir);
	}

	@Override
	public int hashCode() {
		int result = (int) (checkpointId ^ (checkpointId >>> 32));
		result = 31 * result + (int) (timestamp ^ (timestamp >>> 32));
		result = 31 * result + Objects.hashCode(localSnapshotDir);
		return result;
	}

	@Override
	public String toString() {
		return "BackendSnapshotMeta{" +
			", checkpointId=" + checkpointId +
			", timestamp=" + timestamp +
			", localSnapshotDir=" + Objects.toString(localSnapshotDir) +
			'}';
	}
}
