/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.snapshot;

import java.io.Serializable;
import java.util.Objects;

/**
 * Encapsulates meta data for a completed DB snapshot used by backend.
 * This is created by DB and returned to backend.
 */
public class DBSnapshotMeta implements Serializable{

	private static final long serialVersionUID = 6595214484441644131L;

	/**
	 * Path for DB snapshot meta.
	 */
	private final String snapshotMetaPath;

	/**
	 * ID for start region.
	 */
	private final int startRegionId;

	/**
	 * ID for end region.
	 */
	private final int endRegionId;

	/**
	 * The ID of the checkpoint.
	 */
	private final long checkpointId;

	/**
	 * Size of snapshot meta.
	 */
	private final long snapshotMetaSize;

	/**
	 * Incremental size of this snapshot.
	 */
	private final long incrementalSize;

	/**
	 * Full size of this snapshot.
	 */
	private final long fullSize;

	public DBSnapshotMeta(
		long checkpointId,
		int startRegionId,
		int endRegionId,
		long snapshotMetaSize,
		long fullSize,
		long incrementalSize,
		String snapshotMetaPath) {
		this.checkpointId = checkpointId;
		this.startRegionId = startRegionId;
		this.endRegionId = endRegionId;
		this.snapshotMetaSize = snapshotMetaSize;
		this.fullSize = fullSize;
		this.incrementalSize = incrementalSize;
		this.snapshotMetaPath = snapshotMetaPath;
	}

	public String getSnapshotMetaPath() {
		return snapshotMetaPath;
	}

	public int getStartRegionId() {
		return startRegionId;
	}

	public int getEndRegionId() {
		return endRegionId;
	}

	public long getCheckPointId() {
		return checkpointId;
	}

	public long getSnapshotMetaSize() {
		return snapshotMetaSize;
	}

	public long getFullSize() {
		return fullSize;
	}

	public long getIncrementalSize() {
		return incrementalSize;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}

		if (o == null || getClass() != o.getClass()) {
			return false;
		}

		DBSnapshotMeta that = (DBSnapshotMeta) o;

		return checkpointId == that.checkpointId &&
			startRegionId == that.startRegionId &&
			endRegionId == that.endRegionId &&
			snapshotMetaSize == that.snapshotMetaSize &&
			fullSize == that.fullSize &&
			incrementalSize == that.incrementalSize &&
			Objects.equals(snapshotMetaPath, that.snapshotMetaPath);
	}

	@Override
	public int hashCode() {
		int result = startRegionId;
		result = 31 * result + endRegionId;
		result = 31 * result + (int) (checkpointId ^ (checkpointId >>> 32));
		result = 31 * result + (int) (snapshotMetaSize ^ (snapshotMetaSize >>> 32));
		result = 31 * result + (int) (fullSize ^ (fullSize >>> 32));
		result = 31 * result + (int) (incrementalSize ^ (incrementalSize >>> 32));
		result = 31 * result + snapshotMetaPath.hashCode();
		return result;
	}

	@Override
	public String toString() {
		return "DBSnapshotMeta{" +
			", checkpointId=" + checkpointId +
			", startRegionId=" + startRegionId +
			", endRegionId=" + endRegionId +
			", snapshotMetaSize=" + snapshotMetaSize +
			", fullSize=" + fullSize +
			", incrementalSize=" + incrementalSize +
			", snapshotMetaPath=" + snapshotMetaPath +
			'}';
	}
}
