/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.snapshot;

import org.apache.flink.api.common.typeutils.base.TypeSerializerSingleton;
import org.apache.flink.core.memory.DataInputView;
import org.apache.flink.core.memory.DataOutputView;

import java.io.IOException;

/**
 * Serializer for {@link DBSnapshotMeta}.
 */
public class DBSnapshotMetaSerializer extends TypeSerializerSingleton<DBSnapshotMeta> {

	private static final long serialVersionUID = 1L;

	public static final DBSnapshotMetaSerializer INSTANCE = new DBSnapshotMetaSerializer();

	@Override
	public boolean isImmutableType() {
		return true;
	}

	@Override
	public DBSnapshotMeta createInstance() {
		throw new UnsupportedOperationException();
	}

	@Override
	public DBSnapshotMeta copy(DBSnapshotMeta from) {
		return new DBSnapshotMeta(
			from.getCheckPointId(),
			from.getStartRegionId(),
			from.getEndRegionId(),
			from.getSnapshotMetaSize(),
			from.getFullSize(),
			from.getIncrementalSize(),
			from.getSnapshotMetaPath());
	}

	@Override
	public DBSnapshotMeta copy(DBSnapshotMeta from, DBSnapshotMeta reuse) {
		return copy(from);
	}

	@Override
	public int getLength() {
		return -1;
	}

	@Override
	public void serialize(DBSnapshotMeta record, DataOutputView target) throws IOException {
		target.writeLong(record.getCheckPointId());
		target.writeInt(record.getStartRegionId());
		target.writeInt(record.getEndRegionId());
		target.writeLong(record.getSnapshotMetaSize());
		target.writeLong(record.getFullSize());
		target.writeLong(record.getIncrementalSize());
		target.writeUTF(record.getSnapshotMetaPath());
	}

	@Override
	public DBSnapshotMeta deserialize(DataInputView source) throws IOException {
		return new DBSnapshotMeta(source.readLong(), source.readInt(), source.readInt(),
			source.readLong(), source.readLong(), source.readLong(), source.readUTF());
	}

	@Override
	public DBSnapshotMeta deserialize(DBSnapshotMeta reuse, DataInputView source) throws IOException {
		return deserialize(source);
	}

	@Override
	public void copy(DataInputView source, DataOutputView target) throws IOException {
		target.writeInt(source.readInt());
	}

	@Override
	public boolean canEqual(Object obj) {
		return obj instanceof DBSnapshotMeta;
	}

	@Override
	protected boolean isCompatibleSerializationFormatIdentifier(String identifier) {
		return super.isCompatibleSerializationFormatIdentifier(identifier)
			|| identifier.equals(DBSnapshotMeta.class.getCanonicalName());
	}
}
