/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.snapshot;

import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.core.fs.Path;
import org.apache.flink.runtime.state.SnapshotDirectory;
import org.apache.flink.runtime.state.gemini.engine.dbms.GContext;
import org.apache.flink.runtime.state.gemini.engine.fs.FileManager;
import org.apache.flink.util.Preconditions;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.Map;

/**
 * Snapshot operation which will only snapshot dfs.
 */
public class DFSSnapshotOperation extends SnapshotOperation {

	private static final Logger LOG = LoggerFactory.getLogger(DFSSnapshotOperation.class);

	private final FileManager dfsFileManager;

	private SnapshotManager.PendingSnapshot pendingSnapshot;

	public DFSSnapshotOperation(
		GContext gContext,
		SnapshotManager snapshotManager,
		FileManager dfsFileManager) {
		super(gContext, snapshotManager);
		this.dfsFileManager = Preconditions.checkNotNull(dfsFileManager);
	}

	@Override
	public SnapshotManager.PendingSnapshot createPendingSnapshot(
		BackendSnapshotMeta backendSnapshotMeta,
		long accessNumber) {
		long checkpointId = backendSnapshotMeta.getCheckpointId();
		SnapshotCompletableFuture snapshotCompletableFuture = new SnapshotCompletableFuture(gContext,
			gContext.getSupervisor().getSnapshotExecutorGroup());

		Path dfsSnapshotBasePath = dfsFileManager.getBasePath();
		Path dfsMetaPath = ((SnapshotManagerImpl) snapshotManager).getDFSSnapshotMetaPath(
			dfsSnapshotBasePath, checkpointId);

		SnapshotManager.PendingSnapshot pendingSnapshot =
			new SnapshotManager.PendingSnapshot(
				checkpointId,
				backendSnapshotMeta.getTimestamp(),
				snapshotCompletableFuture,
				dfsSnapshotBasePath,
				dfsMetaPath,
				null,
				null,
				accessNumber,
				this);

		snapshotCompletableFuture.setPendingSnapshot(pendingSnapshot);
		this.pendingSnapshot = pendingSnapshot;

		return pendingSnapshot;
	}

	@Override
	public SnapshotManager.PendingSnapshot getPendingSnapshot() {
		return pendingSnapshot;
	}

	@Override
	public DBSnapshotResult getSnapshotResult() throws Exception {
		SnapshotMetaFile.Writer writer = null;
		try {
			writer = SnapshotMetaFile.getWriter(pendingSnapshot.getSnapshotMetaPath());
			RegionSnapshot regionSnapshot = new RegionSnapshot(null, dfsFileManager, writer);

			Map<Integer, Map<Integer, Tuple2<Integer, Long>>> fileMapping = new HashMap<>();
			writePageIndex(pendingSnapshot.getGRegionSnapshotMeta(), null, writer, null,
				regionSnapshot, null, fileMapping);

			pendingSnapshot.setFileMapping(fileMapping);
			writeLocalAndDfsFileMapping(pendingSnapshot, writer, null, dfsFileManager);

			long snapshotMetaSize = writer.getPos();

			SnapshotStat snapshotStat = pendingSnapshot.getSnapshotStat();
			snapshotStat.addAndGetTotalFiles(fileMapping.size());
			snapshotStat.setMetaFileSize(snapshotMetaSize);

			DBSnapshotMeta dbSnapshotMeta = new DBSnapshotMeta(
				pendingSnapshot.getCheckpointId(),
				gContext.getStartRegionId(),
				gContext.getEndRegionId(),
				snapshotMetaSize,
				snapshotStat.addAndGetTotalSize(0),
				snapshotStat.addAndGetIncrementalSize(0),
				writer.getFilePath().toUri().toString());

			writer.close();

			return new DBSnapshotResult(SnapshotDirectory.permanent(pendingSnapshot.getSnapshotBasePath()), dbSnapshotMeta);
		} catch (Exception e) {
			LOG.error("failed to snapshot meta for {}, {}", pendingSnapshot.getCheckpointId(), e.toString());
			closeAndDeleteWriterQuietly(writer);
			throw e;
		}
	}
}
