/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.snapshot;

import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.core.fs.local.LocalFileSystem;
import org.apache.flink.runtime.state.SnapshotDirectory;
import org.apache.flink.runtime.state.gemini.engine.dbms.GContext;
import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiRuntimeException;
import org.apache.flink.runtime.state.gemini.engine.fs.FileManager;
import org.apache.flink.util.Preconditions;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.nio.file.Files;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Snapshot operation which will snapshot local and dfs.
 */
public class LocalAndDFSSnapshotOperation extends SnapshotOperation {

	private static final Logger LOG = LoggerFactory.getLogger(LocalAndDFSSnapshotOperation.class);

	private final FileManager dfsFileManager;

	private final FileManager localFileManager;

	private SnapshotManager.PendingSnapshot pendingSnapshot;

	public LocalAndDFSSnapshotOperation(
		GContext gContext,
		SnapshotManager snapshotManager,
		FileManager dfsFileManager,
		FileManager localFileManager) {
		super(gContext, snapshotManager);
		this.dfsFileManager = Preconditions.checkNotNull(dfsFileManager);
		this.localFileManager = Preconditions.checkNotNull(localFileManager);
	}

	@Override
	public SnapshotManager.PendingSnapshot createPendingSnapshot(
		BackendSnapshotMeta backendSnapshotMeta,
		long accessNumber) {
		long checkpointId = backendSnapshotMeta.getCheckpointId();
		SnapshotCompletableFuture snapshotCompletableFuture = new SnapshotCompletableFuture(gContext,
			gContext.getSupervisor().getSnapshotExecutorGroup());

		Path dfsSnapshotBasePath = dfsFileManager.getBasePath();
		Path dfsMetaPath = ((SnapshotManagerImpl) snapshotManager).getDFSSnapshotMetaPath(
			dfsSnapshotBasePath, checkpointId);
		Path localSnapshotBasePath = backendSnapshotMeta.getLocalSnapshotDir().getDirectory();
		Path localMetaPath = ((SnapshotManagerImpl) snapshotManager).getLocalSnapshotMetaPath(
			localSnapshotBasePath, checkpointId);

		SnapshotManager.PendingSnapshot pendingSnapshot =
			new SnapshotManager.PendingSnapshot(
				checkpointId,
				backendSnapshotMeta.getTimestamp(),
				snapshotCompletableFuture,
				dfsSnapshotBasePath,
				dfsMetaPath,
				localSnapshotBasePath,
				localMetaPath,
				accessNumber,
				this);

		snapshotCompletableFuture.setPendingSnapshot(pendingSnapshot);
		this.pendingSnapshot = pendingSnapshot;

		return pendingSnapshot;
	}

	@Override
	public SnapshotManager.PendingSnapshot getPendingSnapshot() {
		return pendingSnapshot;
	}

	@Override
	public DBSnapshotResult getSnapshotResult() throws Exception {
		SnapshotMetaFile.Writer localWriter = null;
		SnapshotMetaFile.Writer dfsWriter = null;
		try {
			localWriter = SnapshotMetaFile.getWriter(pendingSnapshot.getLocalSnapshotMetaPath());
			RegionSnapshot localRegionSnapshot = new RegionSnapshot(localFileManager, dfsFileManager, localWriter);

			dfsWriter = SnapshotMetaFile.getWriter(pendingSnapshot.getSnapshotMetaPath());
			RegionSnapshot dfsRegionSnapshot = new RegionSnapshot(null, dfsFileManager, dfsWriter);

			// <file id, <group, tuple2<ref, data size>>>
			Map<Integer, Map<Integer, Tuple2<Integer, Long>>> dfsFileMapping = new HashMap<>();
			Map<Integer, Map<Integer, Tuple2<Integer, Long>>> localFileMapping = new HashMap<>();

			writePageIndex(pendingSnapshot.getGRegionSnapshotMeta(), localWriter, dfsWriter,
				localRegionSnapshot, dfsRegionSnapshot, localFileMapping, dfsFileMapping);

			pendingSnapshot.setLocalFileMapping(localFileMapping);
			pendingSnapshot.setFileMapping(dfsFileMapping);

			writeLocalAndDfsFileMapping(pendingSnapshot, localWriter, localFileManager, dfsFileManager);
			writeLocalAndDfsFileMapping(pendingSnapshot, dfsWriter, null, dfsFileManager);

			long localSnapshotMetaSize = localWriter.getPos();
			long dfsSnapshotMetaSize = dfsWriter.getPos();

			SnapshotStat snapshotStat = pendingSnapshot.getSnapshotStat();
			snapshotStat.addAndGetTotalLoalFiles(localFileMapping.size());
			snapshotStat.setLocalMetaFileSize(localSnapshotMetaSize);
			snapshotStat.addAndGetTotalFiles(dfsFileMapping.size());
			snapshotStat.setMetaFileSize(dfsSnapshotMetaSize);

			createHardLinkForLocalFiles(pendingSnapshot.getLocalSnapshotBasePath(), localFileMapping.keySet());

			DBSnapshotMeta localDBSnapshotMeta = new DBSnapshotMeta(
				pendingSnapshot.getCheckpointId(),
				gContext.getStartRegionId(),
				gContext.getEndRegionId(),
				localSnapshotMetaSize,
				snapshotStat.addAndGetTotalLocalSize(0),
				snapshotStat.addAndGetLocalIncrementalSize(0),
				localWriter.getFilePath().toUri().toString());

			DBSnapshotMeta dfsDBSnapshotMeta = new DBSnapshotMeta(
				pendingSnapshot.getCheckpointId(),
				gContext.getStartRegionId(),
				gContext.getEndRegionId(),
				dfsSnapshotMetaSize,
				snapshotStat.addAndGetTotalSize(0),
				snapshotStat.addAndGetIncrementalSize(0),
				dfsWriter.getFilePath().toUri().toString());

			localWriter.close();
			dfsWriter.close();

			return new DBSnapshotResult(
				SnapshotDirectory.permanent(pendingSnapshot.getLocalSnapshotBasePath()),
				localDBSnapshotMeta,
				SnapshotDirectory.permanent(pendingSnapshot.getSnapshotBasePath()),
				dfsDBSnapshotMeta);
		} catch (Exception e) {
			LOG.error("failed to snapshot meta for {}, {}", pendingSnapshot.getCheckpointId(), e.toString());
			closeAndDeleteWriterQuietly(localWriter);
			closeAndDeleteWriterQuietly(dfsWriter);
			throw e;
		}
	}

	@VisibleForTesting
	void createHardLinkForLocalFiles(Path snapshotBasePath, Set<Integer> fileIds) throws Exception {
		FileSystem fileSystem = snapshotBasePath.getFileSystem();
		if (!(fileSystem instanceof LocalFileSystem)) {
			LOG.error("expected LocalFilSystem, actual {}", fileSystem);
			throw new GeminiRuntimeException("hard link only works for local file system, actual " + fileSystem);
		}

		File snapshotBaseDir = new File(snapshotBasePath.toUri());
		for (int fileId : fileIds) {
			File file = new File(localFileManager.getFilePath(fileId));
			try {
				Files.createLink(new File(snapshotBaseDir, file.getName()).toPath(), file.toPath());
			} catch (Exception e) {
				LOG.error("Fail to create hard link from {} to {}, {}", file.getAbsolutePath(), snapshotBasePath.toUri().toString(), e);
				throw e;
			}
		}
	}
}
