/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.snapshot;

import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.runtime.state.gemini.engine.fs.FileManager;
import org.apache.flink.util.Preconditions;

import javax.annotation.Nullable;

import java.util.HashMap;
import java.util.Map;

/**
 * Snapshot for a region.
 */
public class RegionSnapshot {

	@Nullable
	private final FileManager localFileManager;

	private final FileManager dfsFileManager;

	private final SnapshotMetaFile.Writer writer;

	private int region;

	private Map<Integer, Tuple2<Integer, Long>> regionLocalFileMeta;

	private Map<Integer, Tuple2<Integer, Long>> regionDfsFileMeta;

	public RegionSnapshot(
		@Nullable FileManager localFileManager,
		FileManager dfsFileManager,
		SnapshotMetaFile.Writer writer) {
		this.localFileManager = localFileManager;
		this.dfsFileManager = Preconditions.checkNotNull(dfsFileManager);
		this.writer = Preconditions.checkNotNull(writer);
		this.regionLocalFileMeta = new HashMap<>();
		this.regionDfsFileMeta = new HashMap<>();
	}

	public int getRegion() {
		return region;
	}

	public SnapshotMetaFile.Writer getWriter() {
		return writer;
	}

	public Map<Integer, Tuple2<Integer, Long>> getRegionLocalFileMeta() {
		return regionLocalFileMeta;
	}

	public Map<Integer, Tuple2<Integer, Long>> getRegionDfsFileMeta() {
		return regionDfsFileMeta;
	}

	public void updateFileMeta(long address, int reference, long size, boolean isLocal) {
		Map<Integer, Tuple2<Integer, Long>> regionFileMeta =
			isLocal ? regionLocalFileMeta : regionDfsFileMeta;
		int fileId = isLocal ? localFileManager.getFileID(address).get() : dfsFileManager.getFileID(address).get();
		Tuple2<Integer, Long> meta = regionFileMeta.computeIfAbsent(fileId, (no) -> Tuple2.of(0, 0L));
		meta.f0 += reference;
		meta.f1 += size;
	}

	public void reset(int newRegion) {
		region = newRegion;
		if (regionLocalFileMeta != null) {
			regionLocalFileMeta.clear();
		} else {
			regionLocalFileMeta = new HashMap<>();
		}
		if (regionDfsFileMeta != null) {
			regionDfsFileMeta.clear();
		} else {
			regionDfsFileMeta = new HashMap<>();
		}
	}
}
