/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.snapshot;

import org.apache.flink.runtime.state.gemini.engine.dbms.GContext;

import org.apache.flink.shaded.netty4.io.netty.util.concurrent.EventExecutor;
import org.apache.flink.shaded.netty4.io.netty.util.concurrent.EventExecutorGroup;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * An implementation of {@link CompletableFuture} for snapshot.
 */
public class SnapshotCompletableFuture extends CompletableFuture<Boolean> {

	private final GContext gContext;

	private final AtomicInteger runningTasks;

	private final EventExecutorGroup snapshotEventExecutorGroup;

	private final AtomicBoolean endSnapshot;

	private SnapshotManager.PendingSnapshot pendingSnapshot;

	public SnapshotCompletableFuture(
		GContext gContext,
		EventExecutorGroup snapshotEventExecutorGroup) {
		this(gContext, 0, snapshotEventExecutorGroup);
	}

	public SnapshotCompletableFuture(
		GContext gContext,
		int initRunningTasks,
		EventExecutorGroup snapshotEventExecutorGroup) {
		this.gContext = gContext;
		this.runningTasks = new AtomicInteger(initRunningTasks);
		this.snapshotEventExecutorGroup = snapshotEventExecutorGroup;
		this.endSnapshot = new AtomicBoolean(false);
	}

	public void incRunningTask() {
		runningTasks.incrementAndGet();
	}

	public void decRunningTask() {
		int left = runningTasks.decrementAndGet();
		if (left <= 0) {
			for (EventExecutor eventExecutor : snapshotEventExecutorGroup) {
				// TODO #SR break if this future has been canceled/completedExceptionally.
				// make sure all the other places has called the completeExceptionally as needed.
				if (isCancelled() || isCompletedExceptionally()) {
					return;
				}
				try {
					gContext.getSupervisor().getFileCache().sync(eventExecutor);
				} catch (Exception e) {
					// if flush encounters error, then complete exceptionally
					// TODO #SR retry when flush encounters error.
					completeExceptionally(e);
				}
			}
			complete(true);
		}
	}

	public void setPendingSnapshot(SnapshotManager.PendingSnapshot pendingSnapshot) {
		this.pendingSnapshot = pendingSnapshot;
	}

	public SnapshotManager.PendingSnapshot getPendingSnapshot() {
		return this.pendingSnapshot;
	}

	public boolean isEndSnapshot() {
		return endSnapshot.get();
	}

	public void setEndSnapshot() {
		endSnapshot.set(true);
	}
}
