/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.snapshot;

import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.core.fs.Path;
import org.apache.flink.runtime.state.gemini.engine.GRegionID;
import org.apache.flink.runtime.state.gemini.engine.page.PageIndex;

import javax.annotation.Nullable;

import java.io.Closeable;
import java.io.IOException;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;

/**
 * SnapshotManager.
 */
public interface SnapshotManager extends Closeable {

	/**
	 * Whether we need to break the lineage. If we need to break the lineage,
	 * e.g. rescaling case, we would not remember any parental lineage information
	 * and ensure all data must go into newer snapshot to not depend any older snapshots.
	 * In a nutshell, when need to break the lineage, that snapshot cannot be a incremental one.
	 */
	boolean isNeedToBreakLineage();

	void setNeedToBreakLineage(boolean needToBreakLineage);

	/**
	 * synchronously method, start a snapshot, will submit a SnapshotHandler to asynchronously run.
	 */
	void startSnapshot(BackendSnapshotMeta backendSnapshotMeta) throws IOException;

	/**
	 * complete snapshot successfully or exceptionally.
	 * @param throwable The throwable for exceptionally complete, and null for successfully.
	 */
	void endSnapshot(long checkpointId, Throwable throwable);

	/**
	 * finally result of snapshot.
	 */
	PendingSnapshot getPendingSnapshot(long checkpointId);

	/**
	 * Min access number of running snapshot.
	 */
	long getMinRunningSnapshotAccessNumber();

	/**
	 * Get the executorService for snapshot.
	 */
	ExecutorService getSnapshotExecutor();

	/**
	 * This method is called as a notification once a distributed checkpoint has been completed.
	 */
	void notifySnapshotComplete(long snapshotID);

	/**
	 * This method is called as a notification once a distributed checkpoint has been aborted.
	 */
	void notifySnapshotAbort(long snapshotID);

	/**
	 * This method is called as a notification once a distributed checkpoint has been subsumed.
	 */
	void notifySnapshotSubsume(long snapshotID);

	/**
	 * Restore all snapshots according to the snapshot which DB is restored.
	 *
	 * @param snapshotId snapshot which db is restored from.
	 * @param fileMapping file mapping db is restored from.
	 * @param restoredBasePath db base path the snapshot used.
	 * @return all snapshots restored.
	 */
	Map<Long, RestoredSnapshot> restore(long snapshotId, Map<Integer, String> fileMapping, String restoredBasePath);

	/**
	 * snapshot result.
	 */
	class PendingSnapshot {

		long checkpointId;

		long timestamp;

		SnapshotCompletableFuture resultFuture;

		// table name -> region id -> region meta
		private final Map<String, Map<GRegionID, GRegionSnapshotMeta>> regionSnapshotMetas;

		Path snapshotBasePath;

		Path snapshotMetaPath;

		@Nullable
		Path localSnapshotBasePath;

		@Nullable
		Path localSnapshotMetaPath;

		// mapping table for dfs snapshot.
		Map<Integer, Map<Integer, Tuple2<Integer, Long>>> fileMapping;

		// mapping table for local snapshot.
		@Nullable
		Map<Integer, Map<Integer, Tuple2<Integer, Long>>> localFileMapping;

		long accessNumber;

		SnapshotOperation snapshotOperation;

		private final SnapshotStat snapshotStat;

		private boolean isCanceled;

		public PendingSnapshot(
			long checkpointId,
			long timestamp,
			SnapshotCompletableFuture resultFuture,
			Path snapshotBasePath,
			Path snapshotMetaPath,
			Path localSnapshotBasePath,
			Path localSnapshotMetaPath,
			long accessNumber,
			SnapshotOperation snapshotOperation) {
			this.checkpointId = checkpointId;
			this.timestamp = timestamp;
			this.resultFuture = resultFuture;
			this.snapshotBasePath = snapshotBasePath;
			this.snapshotMetaPath = snapshotMetaPath;
			this.localSnapshotBasePath = localSnapshotBasePath;
			this.localSnapshotMetaPath = localSnapshotMetaPath;
			this.accessNumber = accessNumber;
			this.snapshotOperation = snapshotOperation;
			this.regionSnapshotMetas = new ConcurrentHashMap<>();
			this.snapshotStat = new SnapshotStat();
			this.isCanceled = false;
		}

		public long getCheckpointId() {
			return checkpointId;
		}

		public long getTimestamp() {
			return timestamp;
		}

		public SnapshotCompletableFuture getResultFuture() {
			return resultFuture;
		}

		public Path getSnapshotBasePath() {
			return snapshotBasePath;
		}

		public Path getSnapshotMetaPath() {
			return snapshotMetaPath;
		}

		public Path getLocalSnapshotBasePath() {
			return localSnapshotBasePath;
		}

		public Path getLocalSnapshotMetaPath() {
			return localSnapshotMetaPath;
		}

		public long getAccessNumber() {
			return accessNumber;
		}

		public SnapshotOperation getSnapshotOperation() {
			return snapshotOperation;
		}

		public void addGRegionSnapshotMeta(
			String tableName,
			GRegionID regionId,
			PageIndex pageIndex,
			long lastSeqID,
			long removeAllSeqID) {
			regionSnapshotMetas.computeIfAbsent(tableName, (nothing) -> new ConcurrentHashMap<>())
				.put(regionId, new GRegionSnapshotMeta(pageIndex, lastSeqID, removeAllSeqID));
		}

		public Map<String, Map<GRegionID, GRegionSnapshotMeta>> getGRegionSnapshotMeta() {
			return regionSnapshotMetas;
		}

		public void setFileMapping(Map<Integer, Map<Integer, Tuple2<Integer, Long>>> fileMapping) {
			this.fileMapping = fileMapping;
		}

		public Map<Integer, Map<Integer, Tuple2<Integer, Long>>> getFileMapping() {
			return fileMapping;
		}

		public void setLocalFileMapping(Map<Integer, Map<Integer, Tuple2<Integer, Long>>> localFileMapping) {
			this.localFileMapping = localFileMapping;
		}

		public Map<Integer, Map<Integer, Tuple2<Integer, Long>>> getLocalFileMapping() {
			return localFileMapping;
		}

		public SnapshotStat getSnapshotStat() {
			return snapshotStat;
		}

		public void releaseResource() {
			regionSnapshotMetas.clear();
			if (fileMapping != null) {
				fileMapping.clear();
			}
			if (localFileMapping != null) {
				localFileMapping.clear();
			}
		}

		public boolean isCanceled() {
			return isCanceled;
		}

		/**
		 * Should only be called when cancelling this pending snapshot.
		 */
		@VisibleForTesting
		void setCanceled(boolean canceled) {
			isCanceled = canceled;
		}
	}

	/**
	 * Snapshot meta for region.
	 */
	class GRegionSnapshotMeta {

		private PageIndex pageIndex;

		private long lastSeqID;

		private long removeAllSeqID;

		GRegionSnapshotMeta(PageIndex pageIndex, long lastSeqID, long removeAllSeqID) {
			this.pageIndex = pageIndex;
			this.lastSeqID = lastSeqID;
			this.removeAllSeqID = removeAllSeqID;
		}

		public PageIndex getPageIndex() {
			return pageIndex;
		}

		public long getLastSeqID() {
			return lastSeqID;
		}

		public long getRemoveAllSeqID() {
			return removeAllSeqID;
		}
	}

	/**
	 * Completed snapshot information.
	 */
	class CompletedSnapshot {

		private final long checkpointID;

		private final String metaFilePath;

		private final Set<Integer> dataFileIDs;

		public CompletedSnapshot(
			long checkpointID,
			String metaFilePath,
			Set<Integer> dataFileIDs) {
			this.checkpointID = checkpointID;
			this.metaFilePath = metaFilePath;
			this.dataFileIDs = dataFileIDs;
		}

		public long getCheckpointID() {
			return checkpointID;
		}

		public String getMetaFilePath() {
			return metaFilePath;
		}

		public Set<Integer> getDataFileIDs() {
			return dataFileIDs;
		}

		@Override
		public int hashCode() {
			return Objects.hash(checkpointID, metaFilePath, dataFileIDs);
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) {
				return true;
			}
			if (o == null || getClass() != o.getClass()) {
				return false;
			}

			CompletedSnapshot that = (CompletedSnapshot) o;
			return checkpointID == that.checkpointID
				&& metaFilePath.equals(that.metaFilePath)
				&& dataFileIDs.equals(that.dataFileIDs);
		}
	}

	/**
	 * Meta for a restored snapshot.
	 */
	class RestoredSnapshot {

		private final long checkpointID;

		private final String metaFilePath;

		private final Map<Integer, String> fileMapping;

		public RestoredSnapshot(
			long checkpointID,
			String metaFilePath,
			Map<Integer, String> fileMapping) {
			this.checkpointID = checkpointID;
			this.metaFilePath = metaFilePath;
			this.fileMapping = fileMapping;
		}

		public long getCheckpointID() {
			return checkpointID;
		}

		public String getMetaFilePath() {
			return metaFilePath;
		}

		public Map<Integer, String> getFileMapping() {
			return fileMapping;
		}
	}
}
