/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.snapshot;

import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.runtime.state.gemini.engine.fs.GeminiDataInputStream;
import org.apache.flink.runtime.state.gemini.engine.fs.GeminiDataOutputStream;
import org.apache.flink.runtime.state.gemini.engine.fs.GeminiInputStream;
import org.apache.flink.runtime.state.gemini.engine.fs.GeminiOutputStream;
import org.apache.flink.util.Preconditions;

import java.io.IOException;

/**
 * SnapshotMetaFile.
 */
public class SnapshotMetaFile {

	public static Writer getWriter(Path filePath) throws IOException {
		GeminiOutputStream outputStream = new GeminiOutputStream(filePath, FileSystem.WriteMode.OVERWRITE);
		return new WriterImpl(filePath, outputStream);
	}

	public static Reader getReader(Path filePath) throws IOException {
		GeminiInputStream inputStream = new GeminiInputStream(filePath);
		return new ReaderImpl(filePath, inputStream);
	}

	/**
	 * SnapshotMetaFile writer.
	 */
	public abstract static class Writer extends GeminiDataOutputStream {

		private final Path filePath;

		public Writer(
			Path filePath,
			GeminiOutputStream outputStream
		) {
			super(outputStream);

			this.filePath = filePath;
		}

		public Path getFilePath() {
			return filePath;
		}
	}

	/**
	 * Implementation for writer.
	 */
	public static class WriterImpl extends Writer {

		public WriterImpl(
			Path filePath,
			GeminiOutputStream outputStream) {
			super(filePath, outputStream);
		}
	}

	/**
	 * SnapshotMetaFile reader.
	 */
	public abstract static class Reader extends GeminiDataInputStream {

		public Reader(GeminiInputStream inputStream) {
			super(inputStream);
		}
	}

	/**
	 * Implementation for reader.
	 */
	public static class ReaderImpl extends Reader {

		private final Path filePath;

		public ReaderImpl(Path filePath, GeminiInputStream inputStream) throws IOException {
			super(inputStream);

			this.filePath = Preconditions.checkNotNull(filePath);
		}
	}
}
