/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.snapshot;

import org.apache.flink.shaded.guava18.com.google.common.base.MoreObjects;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

/**
 * Statistics for a snapshot.
 */
public class SnapshotStat {

	private AtomicInteger totalPages;

	private AtomicLong totalSize;

	private AtomicInteger incrementalPages;

	private AtomicLong incrementalSize;

	private AtomicInteger totalFiles;

	private long metaFileSize;

	private AtomicInteger totalLocalPages;

	private AtomicLong totalLocalSize;

	private AtomicInteger localIncrementalPages;

	private AtomicLong localIncrementalSize;

	private AtomicInteger totalLocalFiles;

	private long localMetaFileSize;

	private long syncStartTime;

	private long asyncStartTime;

	private long completeTime;

	public SnapshotStat() {
		this.totalPages = new AtomicInteger();
		this.totalSize = new AtomicLong();
		this.incrementalPages = new AtomicInteger();
		this.incrementalSize = new AtomicLong();
		this.totalFiles = new AtomicInteger();
		this.totalLocalPages = new AtomicInteger();
		this.totalLocalSize = new AtomicLong();
		this.localIncrementalPages = new AtomicInteger();
		this.localIncrementalSize = new AtomicLong();
		this.totalLocalFiles = new AtomicInteger();
	}

	public int addAndGetTotalPages(int delta) {
		return totalPages.addAndGet(delta);
	}

	public long addAndGetTotalSize(long delta) {
		return totalSize.addAndGet(delta);
	}

	public int addAndGetIncrementalPages(int delta) {
		return incrementalPages.addAndGet(delta);
	}

	public long addAndGetIncrementalSize(long delta) {
		return incrementalSize.addAndGet(delta);
	}

	public int addAndGetTotalFiles(int delta) {
		return totalFiles.addAndGet(delta);
	}

	public void setMetaFileSize(long size) {
		metaFileSize = size;
	}

	public long getMetaFileSize() {
		return metaFileSize;
	}

	public int addAndGetTotalLocalPages(int delta) {
		return totalLocalPages.addAndGet(delta);
	}

	public long addAndGetTotalLocalSize(long delta) {
		return totalLocalSize.addAndGet(delta);
	}

	public int addAndGetLocalIncrementalPages(int delta) {
		return localIncrementalPages.addAndGet(delta);
	}

	public long addAndGetLocalIncrementalSize(long delta) {
		return localIncrementalSize.addAndGet(delta);
	}

	public int addAndGetTotalLoalFiles(int delta) {
		return totalLocalFiles.addAndGet(delta);
	}

	public void setLocalMetaFileSize(long size) {
		localMetaFileSize = size;
	}

	public long getLocalMetaFileSize() {
		return localMetaFileSize;
	}

	public void setSyncStartTime(long time) {
		syncStartTime = time;
	}

	public long getSyncStartTime() {
		return syncStartTime;
	}

	public void setAsyncStartTime(long time) {
		asyncStartTime = time;
	}

	public long getAsyncStartTime() {
		return asyncStartTime;
	}

	public void setCompleteTime(long time) {
		completeTime = time;
	}

	public long getCompleteTime() {
		return completeTime;
	}

	@Override
	public String toString() {
		return MoreObjects.toStringHelper(this).
			add("totalPages", totalPages.get()).
			add("totalSize", totalSize.get()).
			add("incrementalPages", incrementalPages.get()).
			add("incrementalSize", incrementalSize.get()).
			add("totalFiles", totalFiles.get()).
			add("metaFileSize", metaFileSize).
			add("totalLocalPages", totalLocalPages.get()).
			add("totalLocalSize", totalLocalSize.get()).
			add("localIncrementalPages", localIncrementalPages.get()).
			add("localIncrementalSize", localIncrementalSize.get()).
			add("totalLocalFiles", totalLocalFiles.get()).
			add("localMetaFileSize", localMetaFileSize).
			add("syncStartTime", syncStartTime).
			add("asyncStartTime", asyncStartTime).
			add("completeTime", completeTime).
			add("totalSyncTime", (asyncStartTime - syncStartTime) + "ms").
			add("totalAsyncTime", (completeTime - asyncStartTime) + "ms").
			add("totalTime", (completeTime - syncStartTime) + "ms").
			toString();
	}
}
