/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.utils;

import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.typeutils.CompatibilityResult;
import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.api.common.typeutils.TypeSerializerConfigSnapshot;
import org.apache.flink.core.memory.DataInputView;
import org.apache.flink.core.memory.DataOutputView;
import org.apache.flink.runtime.memory.AbstractPagedInputView;
import org.apache.flink.runtime.memory.AbstractPagedOutputView;
import org.apache.flink.util.Preconditions;

import java.io.IOException;

/**
 * A thread local serializer which can be concurrently accessed.
 */
public class ThreadLocalTypeSerializer<T, S extends TypeSerializer<T>> extends TypeSerializer<T> {

	private final S originSerializer;

	private final ThreadLocal<S> threadLocalSerializer;

	public ThreadLocalTypeSerializer(S serializer) {
		this.originSerializer = Preconditions.checkNotNull(serializer);
		this.threadLocalSerializer = new ThreadLocal<>();
		// not duplicate the serializer for the thread where
		// this thread local serializer is created
		this.threadLocalSerializer.set(serializer);
	}

	public static <T, S extends TypeSerializer<T>> ThreadLocalTypeSerializer<T, S> of(S serializer) {
		return new ThreadLocalTypeSerializer<>(serializer);
	}

	public S getOriginSerializer() {
		return originSerializer;
	}

	@VisibleForTesting
	S getSerializer() {
		S serializer = threadLocalSerializer.get();
		if (serializer == null) {
			serializer = (S) originSerializer.duplicate();
			threadLocalSerializer.set(serializer);
		}
		return serializer;
	}

	@Override
	public boolean isImmutableType() {
		return originSerializer.isImmutableType();
	}

	@Override
	public TypeSerializer<T> duplicate() {
		throw new UnsupportedOperationException("serializer duplication should not happen in GeminiDB");
	}

	@Override
	public T createInstance() {
		return getSerializer().createInstance();
	}

	@Override
	public T copy(T from) {
		return getSerializer().copy(from);
	}

	@Override
	public T copy(T from, T reuse) {
		return getSerializer().copy(from, reuse);
	}

	@Override
	public int getLength() {
		return originSerializer.getLength();
	}

	@Override
	public void serialize(T record, DataOutputView target) throws IOException {
		getSerializer().serialize(record, target);
	}

	@Override
	public T deserialize(DataInputView source) throws IOException {
		return getSerializer().deserialize(source);
	}

	@Override
	public T deserialize(T reuse, DataInputView source) throws IOException {
		return getSerializer().deserialize(reuse, source);
	}

	@Override
	public int serializeToPages(T record, AbstractPagedOutputView target) throws IOException {
		return getSerializer().serializeToPages(record, target);
	}

	@Override
	public T deserializeFromPages(AbstractPagedInputView source) throws IOException {
		return getSerializer().deserializeFromPages(source);
	}

	@Override
	public T deserializeFromPages(T reuse, AbstractPagedInputView source) throws IOException {
		return getSerializer().deserializeFromPages(reuse, source);
	}

	@Override
	public T mapFromPages(AbstractPagedInputView source) throws IOException {
		return getSerializer().mapFromPages(source);
	}

	@Override
	public T mapFromPages(T reuse, AbstractPagedInputView source) throws IOException {
		return getSerializer().mapFromPages(reuse, source);
	}

	@Override
	public void copy(DataInputView source, DataOutputView target) throws IOException {
		getSerializer().copy(source, target);
	}

	@Override
	public boolean equals(Object obj) {
		if (obj instanceof ThreadLocalTypeSerializer) {
			ThreadLocalTypeSerializer<?, ?> other = (ThreadLocalTypeSerializer<?, ?>) obj;

			return other.canEqual(this) && other.originSerializer.equals(originSerializer);
		} else {
			return false;
		}
	}

	@Override
	public boolean canEqual(Object obj) {
		return obj instanceof ThreadLocalTypeSerializer;
	}

	@Override
	public int hashCode() {
		return originSerializer.hashCode();
	}

	@Override
	public TypeSerializerConfigSnapshot snapshotConfiguration() {
		throw new UnsupportedOperationException("thread local serializer is only used in GeminiDB");
	}

	@Override
	public CompatibilityResult<T> ensureCompatibility(TypeSerializerConfigSnapshot configSnapshot) {
		throw new UnsupportedOperationException("thread local serializer is only used in GeminiDB");
	}
}
