/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.vm;

import org.apache.flink.metrics.Gauge;
import org.apache.flink.metrics.MetricGroup;
import org.apache.flink.runtime.state.gemini.engine.metrics.MetricsRegisterAble;

import org.apache.flink.shaded.guava18.com.google.common.base.MoreObjects;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;

import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.AVERAGE_MAJOR_COMPACTION_PAGE_NUM;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.AVERAGE_MINOR_COMPACTION_PAGE_NUM;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.AVERAGE_PAGE_CHAIN_LEN;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.AVERAGE_PAGE_SIZE;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.TOTAL_EVICT_BLOCK_COUNT;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.TOTAL_INDEX_CAPACITY;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.TOTAL_LOGIC_PAGE_COUNT;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.TOTAL_LOGIC_PAGE_SIZE;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.TOTAL_PAGE_CACHE_HIT_COUNT;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.TOTAL_PAGE_CACHE_LRU_HIT_COUNT;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.TOTAL_PAGE_CACHE_MISS_COUNT;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.TOTAL_PAGE_EVICT_SIZE;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.TOTAL_PAGE_USED_MEMORY;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.TOTAL_RUNNING_MARJOR_COMPACTION;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.TOTAL_RUNNING_MINOR_COMPACTION;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.TOTAL_WR_BUFFER_HIT_COUNT;
import static org.apache.flink.runtime.state.gemini.engine.metrics.CacheMetrics.TOTAL_WR_BUFFER_MISS_COUNT;

/**
 * CacheStats.
 */
public class CacheStats implements MetricsRegisterAble {

	private AtomicLong totalLogicPageSize = new AtomicLong(0);
	private AtomicLong totalPageUsedMemory = new AtomicLong(0);
	private AtomicLong totalIndexCapacity = new AtomicLong(0);
	private AtomicLong totalLogicPageCount = new AtomicLong(0);
	private AtomicLong totalWriteBufferHitCount = new AtomicLong(0);
	private AtomicLong totalWriteBufferMissCount = new AtomicLong(0);
	private AtomicLong totalPageCacheHitCount = new AtomicLong(0);
	private AtomicLong totalPageCacheLRUHitCount = new AtomicLong(0);
	private AtomicLong totalPageCacheMissCount = new AtomicLong(0);
	private AtomicLong totalPageCacheEvictSize = new AtomicLong(0);
	private AtomicLong totalPageCacheForceEvictSize = new AtomicLong(0);
	private AtomicLong totalPageFlushToLocalSize = new AtomicLong(0);
	private AtomicLong totalPageFlushToDFSSize = new AtomicLong(0);
	private AtomicLong totalPageCachePrefetchSize = new AtomicLong(0);
	private AtomicLong totalPageForceFetchByCompactionCount = new AtomicLong(0);
	private AtomicLong totalPageFetchInSnapshotSize = new AtomicLong(0);
	private AtomicLong totalLogicPageChainLen = new AtomicLong(0);
	private AtomicInteger totalRunningMinorCompaction = new AtomicInteger(0);
	private AtomicInteger totalRunningMajorCompaction = new AtomicInteger(0);
	private AtomicInteger totalRunningMinorCompactionByRead = new AtomicInteger(0);
	private final AtomicLong totalMinorCompactions = new AtomicLong(0);
	private final AtomicLong totalMinorCompactedPages = new AtomicLong(0);
	private final AtomicLong totalMajorCompactions = new AtomicLong(0);
	private final AtomicLong totalMajorCompactedPages = new AtomicLong(0);
	private final AtomicLong totalEvictBlockCount = new AtomicLong(0);

	public void addTotalLogicPageSize(int pageSize) {
		this.totalLogicPageSize.addAndGet(pageSize);
	}

	public void addPageUsedMemory(int pageSize) {
		this.totalPageUsedMemory.addAndGet(pageSize);
	}

	public void addIndexCapacity(int indexCapacity) {
		this.totalIndexCapacity.addAndGet(indexCapacity);
	}

	public long getTotalLogicPageSize() {
		return totalLogicPageSize.get();
	}

	public long getTotalPageUsedMemory() {
		return this.totalPageUsedMemory.get();
	}

	public long getTotalIndexCapacity() {
		return this.totalIndexCapacity.get();
	}

	public void addWriteBufferHitCount() {
		totalWriteBufferHitCount.incrementAndGet();
	}

	public void addWriteBufferMissCount() {
		totalWriteBufferMissCount.incrementAndGet();
	}

	public void addPageCacheHitCount() {
		totalPageCacheHitCount.incrementAndGet();
	}

	public void addPageCacheMissCount() {
		totalPageCacheMissCount.incrementAndGet();
	}

	public void addPageForceFetchByCompactionCount() {
		totalPageForceFetchByCompactionCount.incrementAndGet();
	}

	public void addPageCacheEvictSize(int size) {
		totalPageCacheEvictSize.addAndGet(size);
	}

	public void addPageCacheForceEvictSize(int size) {
		totalPageCacheForceEvictSize.addAndGet(size);
	}

	public void addPageCacheFlushSize(int size, boolean isLocal) {
		if (isLocal) {
			totalPageFlushToLocalSize.addAndGet(size);
		} else {
			totalPageFlushToDFSSize.addAndGet(size);
		}
	}

	public void addTotalPageFetchInSnapshotSize(int size) {
		totalPageFetchInSnapshotSize.addAndGet(size);
	}

	@Override
	public String toString() {
		return MoreObjects.toStringHelper(this).
			add(TOTAL_PAGE_USED_MEMORY, totalPageUsedMemory).
			add(TOTAL_LOGIC_PAGE_SIZE, totalLogicPageSize).
			add(TOTAL_INDEX_CAPACITY, totalIndexCapacity).
			add(TOTAL_LOGIC_PAGE_COUNT, totalLogicPageCount).
			add("totalLogicPageChainLen", totalLogicPageChainLen.get()).
			add(TOTAL_WR_BUFFER_HIT_COUNT, totalWriteBufferHitCount).
			add(TOTAL_WR_BUFFER_MISS_COUNT, totalWriteBufferMissCount).
			add(TOTAL_PAGE_CACHE_HIT_COUNT, totalPageCacheHitCount).
			add(TOTAL_PAGE_CACHE_LRU_HIT_COUNT, totalPageCacheLRUHitCount).
			add(TOTAL_PAGE_CACHE_MISS_COUNT, totalPageCacheMissCount).
			add(TOTAL_PAGE_EVICT_SIZE, totalPageCacheEvictSize).
			add("totalPageCacheForceEvictSize", totalPageCacheForceEvictSize).
			add("totalPageFlushToLocalSize", totalPageFlushToLocalSize).
			add("totalPageFlushToDFSSize", totalPageFlushToDFSSize).
			add("totalPageFetchInSnapshotSize", totalPageFetchInSnapshotSize).
			add("totalPageCachePrefetchSize", totalPageCachePrefetchSize).
			add("totalPageForceFetchByCompactionCount", totalPageForceFetchByCompactionCount).
			add(TOTAL_RUNNING_MARJOR_COMPACTION, totalRunningMajorCompaction.get()).
			add(TOTAL_RUNNING_MINOR_COMPACTION, totalRunningMinorCompaction.get()).
			add("totalRunningMinorCompactionByRead", totalRunningMinorCompactionByRead.get()).
			toString();
	}

	public void addTotalLogicPageCount(int logicPageCount) {
		totalLogicPageCount.addAndGet(logicPageCount);
	}

	public void addTotalLogicPageChainLen(int logicPageChainLen) {
		totalLogicPageChainLen.addAndGet(logicPageChainLen);
	}

	public void addTotalRuningMajorCompactedPages(int i) {
		this.totalRunningMajorCompaction.addAndGet(i);
	}

	public void addTotalRuningMinorCompactedPages(int i) {
		this.totalRunningMinorCompaction.addAndGet(i);
	}

	public int getRuningMajorCompactedPages() {
		return totalRunningMajorCompaction.get();
	}

	public int getRuningMinorCompactedPages() {
		return totalRunningMinorCompaction.get();
	}

	public void addRuningMinorCompactionByRead(int i) {
		this.totalRunningMinorCompactionByRead.addAndGet(i);
	}

	public int getRuningMinorCompactionByRead() {
		return totalRunningMinorCompactionByRead.get();
	}

	public void addPageCacheLRUHitCount() {
		totalPageCacheLRUHitCount.incrementAndGet();
	}

	public void addMinorCompactedPages(int n) {
		this.totalMinorCompactions.addAndGet(1);
		this.totalMinorCompactedPages.addAndGet(n);
	}

	public void addMajorCompactedPages(int n) {
		this.totalMajorCompactions.addAndGet(1);
		this.totalMajorCompactedPages.addAndGet(n);
	}

	private double getAveragePageChainLen() {
		long curLogicPageCount = totalLogicPageCount.get();
		return curLogicPageCount == 0 ? 0 : ((double) totalLogicPageChainLen.get()) / curLogicPageCount;
	}

	private double getAveragePageSize() {
		long curLogicPageChainLen = totalLogicPageChainLen.get();
		return curLogicPageChainLen == 0 ? 0 : ((double) totalLogicPageSize.get()) / curLogicPageChainLen;
	}

	private double getAverageMinorCompactionPageNum() {
		long curMinorCompactions = totalMinorCompactions.get();
		return curMinorCompactions == 0 ? 0 : ((double) totalMinorCompactedPages.get()) / curMinorCompactions;
	}

	private double getAverageMajorCompactionPageNum() {
		long curMajorCompactions = totalMajorCompactions.get();
		return curMajorCompactions == 0 ? 0 : ((double) totalMajorCompactedPages.get()) / curMajorCompactions;
	}

	public void addEvictBlock(int n) {
		this.totalEvictBlockCount.addAndGet(n);
	}

	@Override
	public void registerMetrics(MetricGroup metricGroup) {
		metricGroup.gauge(TOTAL_PAGE_USED_MEMORY, (Gauge<Long>) () -> totalPageUsedMemory.get());
		metricGroup.gauge(TOTAL_LOGIC_PAGE_SIZE, (Gauge<Long>) () -> totalLogicPageSize.get());
		metricGroup.gauge(TOTAL_INDEX_CAPACITY, (Gauge<Long>) () -> totalIndexCapacity.get());
		metricGroup.gauge(TOTAL_LOGIC_PAGE_COUNT, (Gauge<Long>) () -> totalLogicPageCount.get());
		metricGroup.gauge(AVERAGE_PAGE_CHAIN_LEN, (Gauge<Double>) () -> getAveragePageChainLen());
		metricGroup.gauge(AVERAGE_PAGE_SIZE, (Gauge<Double>) () -> getAveragePageSize());
		metricGroup.gauge(AVERAGE_MINOR_COMPACTION_PAGE_NUM, (Gauge<Double>) () -> getAverageMinorCompactionPageNum());
		metricGroup.gauge(AVERAGE_MAJOR_COMPACTION_PAGE_NUM, (Gauge<Double>) () -> getAverageMajorCompactionPageNum());
		metricGroup.gauge(TOTAL_WR_BUFFER_HIT_COUNT, (Gauge<Long>) () -> totalWriteBufferHitCount.get());
		metricGroup.gauge(TOTAL_WR_BUFFER_MISS_COUNT, (Gauge<Long>) () -> totalWriteBufferMissCount.get());
		metricGroup.gauge(TOTAL_PAGE_CACHE_HIT_COUNT, (Gauge<Long>) () -> totalPageCacheHitCount.get());
		metricGroup.gauge(TOTAL_PAGE_CACHE_MISS_COUNT, (Gauge<Long>) () -> totalPageCacheMissCount.get());
		metricGroup.gauge(TOTAL_PAGE_CACHE_LRU_HIT_COUNT, (Gauge<Long>) () -> totalPageCacheLRUHitCount.get());
		metricGroup.gauge(TOTAL_PAGE_EVICT_SIZE, (Gauge<Long>) () -> totalPageCacheEvictSize.get());
		metricGroup.gauge(TOTAL_RUNNING_MARJOR_COMPACTION, (Gauge<Integer>) () -> totalRunningMajorCompaction.get());
		metricGroup.gauge(TOTAL_RUNNING_MINOR_COMPACTION,
			(Gauge<Integer>) () -> totalRunningMinorCompaction.get() + totalRunningMinorCompactionByRead.get());
		metricGroup.gauge(TOTAL_EVICT_BLOCK_COUNT, (Gauge<Long>) () -> totalEvictBlockCount.get());
	}
}
