/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.engine.vm;

import org.apache.flink.runtime.state.gemini.engine.exceptions.GeminiRuntimeException;
import org.apache.flink.runtime.state.gemini.engine.page.DataPage;
import org.apache.flink.runtime.state.gemini.engine.rm.ReferenceCount.ReleaseType;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

/**
 * FutureDataPage.
 */
public class FutureDataPage {
	private final int size;
	private final CompletableFuture<DataPage> dataPageCompletedFuture;
	private volatile boolean removed = false;

	public FutureDataPage(int size) {
		this.size = size;
		this.dataPageCompletedFuture = new CompletableFuture<>();
	}

	public FutureDataPage(
		DataPage dataPage) {
		this.size = dataPage.getSize();
		this.dataPageCompletedFuture = CompletableFuture.completedFuture(dataPage);
	}

	public int getSize() {
		return this.size;
	}

	public boolean complete(DataPage dataPage) {
		boolean result = this.dataPageCompletedFuture.complete(dataPage);
		if (result && removed) {
			dataPage.delReferenceCount(ReleaseType.Normal);
		}
		return result;
	}

	public void completeExceptionally(Throwable throwable) {
		this.dataPageCompletedFuture.completeExceptionally(throwable);
	}

	public DataPage get() {
		try {
			return this.dataPageCompletedFuture.get();
		} catch (Exception e) {
			throw new GeminiRuntimeException("FutureDataPage error:" + e.getMessage(), e);
		}
	}

	public DataPage get(int timeout) {
		try {
			return this.dataPageCompletedFuture.get(timeout, TimeUnit.MILLISECONDS);
		} catch (Exception e) {
			throw new GeminiRuntimeException("FutureDataPage error, Msg=" + e.getMessage() + " ,cause=" + e.getClass().getSimpleName(),
				e);
		}
	}

	public boolean isFail() {
		return dataPageCompletedFuture.isCompletedExceptionally();
	}

	public boolean isDone() {
		return dataPageCompletedFuture.isDone();
	}

	public void removed() {
		removed = true;
		if (isDone() && !isFail()) {
			try {
				DataPage dataPage = dataPageCompletedFuture.get();
				dataPage.delReferenceCount(ReleaseType.Normal);
			} catch (Exception e) {
				throw new GeminiRuntimeException(e);
			}
		}
	}
}
