/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.internal;

import org.apache.flink.runtime.state.IncrementalStateHandle;
import org.apache.flink.runtime.state.KeyGroupRange;
import org.apache.flink.runtime.state.KeyedStateHandle;
import org.apache.flink.runtime.state.StreamStateHandle;
import org.apache.flink.runtime.state.gemini.GeminiInternalStateBackend;
import org.apache.flink.runtime.state.gemini.engine.snapshot.DBSnapshotMeta;
import org.apache.flink.util.Preconditions;

/**
 * An base implementation of {@link KeyedStateHandle} for {@link GeminiInternalStateBackend}.
 */
public abstract class AbstractGeminiKeyedStateHandle implements KeyedStateHandle, IncrementalStateHandle {

	private static final long serialVersionUID = -7957128385176716193L;

	/**
	 * Id of the checkpoint that created this state handle.
	 */
	protected final long checkpointId;

	/**
	 * The key-group range covered by this state handle.
	 */
	protected final KeyGroupRange keyGroupRange;

	/**
	 * Meta data state of the checkpoint.
	 */
	protected final StreamStateHandle metaStateHandle;

	/**
	 * The DB snapshot directory state handle.
	 */
	protected final DirectoryStreamStateHandle dbSnapshotDirectoryHandle;

	/**
	 * Meta for DB snapshot result.
	 */
	protected final DBSnapshotMeta dbSnapshotMeta;

	public AbstractGeminiKeyedStateHandle(
		final long checkpointId,
		final KeyGroupRange keyGroupRange,
		final StreamStateHandle metaStateHandle,
		final DirectoryStreamStateHandle dbSnapshotDirectoryHandle,
		final DBSnapshotMeta dbSnapshotMeta
	) {
		this.checkpointId = checkpointId;
		this.keyGroupRange = Preconditions.checkNotNull(keyGroupRange);
		this.metaStateHandle = Preconditions.checkNotNull(metaStateHandle);
		this.dbSnapshotDirectoryHandle = Preconditions.checkNotNull(dbSnapshotDirectoryHandle);
		this.dbSnapshotMeta = Preconditions.checkNotNull(dbSnapshotMeta);
	}

	@Override
	public KeyGroupRange getKeyGroupRange() {
		return keyGroupRange;
	}

	public long getCheckpointId() {
		return checkpointId;
	}

	public StreamStateHandle getMetaStateHandle() {
		return metaStateHandle;
	}

	public DirectoryStreamStateHandle getDBSnapshotDirectoryHandle() {
		return dbSnapshotDirectoryHandle;
	}

	public DBSnapshotMeta getDBSnapshotMeta() {
		return dbSnapshotMeta;
	}

	@Override
	public long getStateSize() {
		return dbSnapshotMeta.getSnapshotMetaSize() + dbSnapshotMeta.getIncrementalSize();
	}

	@Override
	public long getFullStateSize() {
		return dbSnapshotMeta.getSnapshotMetaSize() + dbSnapshotMeta.getFullSize();
	}
}
