/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.internal;

import org.apache.flink.core.fs.FSDataInputStream;
import org.apache.flink.core.fs.Path;
import org.apache.flink.core.memory.DataInputViewStreamWrapper;
import org.apache.flink.core.memory.DataOutputViewStreamWrapper;
import org.apache.flink.runtime.state.KeyGroupRange;
import org.apache.flink.runtime.state.KeyedStateHandle;
import org.apache.flink.runtime.state.SharedStateRegistry;
import org.apache.flink.runtime.state.StreamStateHandle;
import org.apache.flink.runtime.state.gemini.engine.snapshot.DBSnapshotMeta;
import org.apache.flink.runtime.state.gemini.engine.snapshot.DBSnapshotMetaSerializer;
import org.apache.flink.util.Preconditions;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.Objects;

/**
 * A state handle used by JM.
 */
public class GeminiKeyedStateHandle extends AbstractGeminiKeyedStateHandle implements StreamStateHandle {

	private static final Logger LOG = LoggerFactory.getLogger(GeminiKeyedStateHandle.class);

	private static final long serialVersionUID = 1L;

	/**
	 * Once the shared states are registered, it is the {@link SharedStateRegistry}'s
	 * responsibility to cleanup those shared states.
	 * But in the cases where the state handle is discarded before performing the registration,
	 * the handle should delete all the shared states created by it.
	 *
	 * <p>his variable is not null iff the handles was registered.
	 */
	private transient SharedStateRegistry sharedStateRegistry;

	public GeminiKeyedStateHandle(
		final long checkpointId,
		final KeyGroupRange keyGroupRange,
		final StreamStateHandle metaStateHandle,
		final DirectoryStreamStateHandle dbSnapshotDirectoryHandle,
		final DBSnapshotMeta dbSnapshotMeta
	) {
		super(checkpointId, keyGroupRange, metaStateHandle, dbSnapshotDirectoryHandle, dbSnapshotMeta);
	}

	@Override
	public KeyedStateHandle getIntersection(KeyGroupRange otherKeyGroupRange) {
		return KeyGroupRange.EMPTY_KEY_GROUP_RANGE.equals(
			this.keyGroupRange.getIntersection(otherKeyGroupRange)) ? null : this;
	}

	@Override
	public void registerSharedStates(SharedStateRegistry stateRegistry) {
		// This is a quick check to avoid that we register twice with the same registry. However, the code allows to
		// register again with a different registry. The implication is that ownership is transferred to this new
		// registry. This should only happen in case of a restart, when the CheckpointCoordinator creates a new
		// SharedStateRegistry for the current attempt and the old registry becomes meaningless. We also assume that
		// an old registry object from a previous run is due to be GCed and will never be used for registration again.
		Preconditions.checkState(
			sharedStateRegistry != stateRegistry,
			"The state handle has already registered its shared states to the given registry.");

		sharedStateRegistry = Preconditions.checkNotNull(stateRegistry);

		LOG.trace("Registering GeminiKeyedStateHandle for checkpoint {} from backend.", checkpointId);

		// TODO how to deal with dbSnapshotDirectoryHandle for rescaling
	}

	@Override
	public void discardState() throws Exception {
		SharedStateRegistry registry = this.sharedStateRegistry;
		final boolean isRegistered = (registry != null);

		LOG.trace("Discarding GeminiKeyedStateHandle (registered = {}) for checkpoint {} from backend.",
			isRegistered,
			checkpointId);

		try {
			metaStateHandle.discardState();
		} catch (Exception e) {
			LOG.warn("Could not properly discard meta data.", e);
		}

		// TODO how to deal with dbSnapshotDirectoryHandle for rescaling
	}

	@Override
	public FSDataInputStream openInputStream() throws IOException {
		Preconditions.checkNotNull(metaStateHandle, "metaStateHandle is null");
		return metaStateHandle.openInputStream();
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}

		if (o == null || getClass() != o.getClass()) {
			return false;
		}

		GeminiKeyedStateHandle that = (GeminiKeyedStateHandle) o;

		return checkpointId == that.checkpointId &&
			Objects.equals(keyGroupRange, that.keyGroupRange) &&
			Objects.equals(metaStateHandle, that.metaStateHandle) &&
			Objects.equals(dbSnapshotDirectoryHandle, that.dbSnapshotDirectoryHandle) &&
			Objects.equals(dbSnapshotMeta, that.dbSnapshotMeta);
	}

	@Override
	public int hashCode() {
		int result = Objects.hashCode(keyGroupRange);
		result = 31 * result + (int) (getCheckpointId() ^ (getCheckpointId() >>> 32));
		result = 31 * result + Objects.hashCode(metaStateHandle);
		result = 31 * result + Objects.hashCode(dbSnapshotDirectoryHandle);
		result = 31 * result + Objects.hashCode(dbSnapshotMeta);
		return result;
	}

	@Override
	public String toString() {
		return "GeminiKeyedStateHandle {" +
			", checkpointId=" + checkpointId +
			", keyGroupRange=" + keyGroupRange +
			", metaStateHandle=" + metaStateHandle +
			", dbSnapshotDirectoryHandle=" + dbSnapshotDirectoryHandle +
			", dbSnapshotMeta=" + dbSnapshotMeta +
			"}";
	}

	public static void serializeDBPart(
		GeminiKeyedStateHandle stateHandle,
		DataOutputStream dos) throws IOException {
		String dbSnapshotDirectory =
			stateHandle.getDBSnapshotDirectoryHandle().getDirectory().toUri().toString();
		dos.writeUTF(dbSnapshotDirectory);

		DBSnapshotMeta dbSnapshotMeta = stateHandle.getDBSnapshotMeta();
		DBSnapshotMetaSerializer.INSTANCE.serialize(dbSnapshotMeta, new DataOutputViewStreamWrapper(dos));
	}

	public static GeminiKeyedStateHandle deserializeDBPartAndCreateStateHandle(
		long checkpointId,
		KeyGroupRange keyGroupRange,
		StreamStateHandle metaStateHandle,
		DataInputStream dis) throws IOException {

		String dbSnapshotDirectory = dis.readUTF();
		DBSnapshotMeta dbSnapshotMeta =
			DBSnapshotMetaSerializer.INSTANCE.deserialize(new DataInputViewStreamWrapper(dis));

		return new GeminiKeyedStateHandle(
			checkpointId,
			keyGroupRange,
			metaStateHandle,
			new DirectoryStreamStateHandle(new Path(dbSnapshotDirectory)),
			dbSnapshotMeta);
	}
}
