/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.internal;

import org.apache.flink.runtime.state.KeyGroupRange;
import org.apache.flink.runtime.state.KeyedStateHandle;
import org.apache.flink.runtime.state.SharedStateRegistry;
import org.apache.flink.runtime.state.StreamStateHandle;
import org.apache.flink.runtime.state.gemini.engine.snapshot.DBSnapshotMeta;
import org.apache.flink.util.ExceptionUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Objects;

/**
 * A state handle used for local recovery.
 */
public class GeminiLocalKeyedStateHandle extends AbstractGeminiKeyedStateHandle {

	private static final Logger LOG = LoggerFactory.getLogger(GeminiLocalKeyedStateHandle.class);

	private static final long serialVersionUID = 1L;

	public GeminiLocalKeyedStateHandle(
		final long checkpointId,
		final KeyGroupRange keyGroupRange,
		final StreamStateHandle metaStateHandle,
		final DirectoryStreamStateHandle dbSnapshotDirectoryHandle,
		final DBSnapshotMeta dbSnapshotMeta
	) {
		super(checkpointId, keyGroupRange, metaStateHandle, dbSnapshotDirectoryHandle, dbSnapshotMeta);
	}

	@Override
	public KeyedStateHandle getIntersection(KeyGroupRange otherKeyGroupRange) {
		return this.keyGroupRange.getIntersection(otherKeyGroupRange).getNumberOfKeyGroups() > 0 ? this : null;
	}

	@Override
	public void registerSharedStates(SharedStateRegistry stateRegistry) {
		// Nothing to do, this is for local use only.
	}

	@Override
	public void discardState() throws Exception {
		Exception collectedEx = null;

		try {
			metaStateHandle.discardState();
		} catch (Exception e) {
			collectedEx = e;
		}

		try {
			dbSnapshotDirectoryHandle.discardState();
		} catch (Exception e) {
			collectedEx = ExceptionUtils.firstOrSuppressed(e, collectedEx);
		}

		if (collectedEx != null) {
			throw collectedEx;
		}
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}

		if (o == null || getClass() != o.getClass()) {
			return false;
		}

		GeminiLocalKeyedStateHandle that = (GeminiLocalKeyedStateHandle) o;

		return checkpointId == that.checkpointId &&
			Objects.equals(keyGroupRange, that.keyGroupRange) &&
			Objects.equals(metaStateHandle, that.metaStateHandle) &&
			Objects.equals(dbSnapshotDirectoryHandle, that.dbSnapshotDirectoryHandle) &&
			Objects.equals(dbSnapshotMeta, that.dbSnapshotMeta);
	}

	@Override
	public int hashCode() {
		int result = Objects.hashCode(keyGroupRange);
		result = 31 * result + (int) (getCheckpointId() ^ (getCheckpointId() >>> 32));
		result = 31 * result + Objects.hashCode(metaStateHandle);
		result = 31 * result + Objects.hashCode(dbSnapshotDirectoryHandle);
		result = 31 * result + Objects.hashCode(dbSnapshotMeta);
		return result;
	}

	@Override
	public String toString() {
		return "GeminiLocalKeyedStateHandle {" +
			", checkpointId=" + checkpointId +
			", keyGroupRange=" + keyGroupRange +
			", metaStateHandle=" + metaStateHandle +
			", dbSnapshotDirectoryHandle=" + dbSnapshotDirectoryHandle +
			", dbSnapshotMeta=" + dbSnapshotMeta +
			"}";
	}
}
