/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.keyed;

import org.apache.flink.runtime.state.StateStorage;
import org.apache.flink.runtime.state.gemini.engine.GRegion;
import org.apache.flink.runtime.state.gemini.engine.hashtable.AbstractGRegionKMapImpl;
import org.apache.flink.runtime.state.gemini.engine.hashtable.AbstractGTableKeyedMapImpl;
import org.apache.flink.runtime.state.gemini.engine.hashtable.GRegionKMapImpl;
import org.apache.flink.runtime.state.keyed.AbstractKeyedMapState;
import org.apache.flink.runtime.state.keyed.KeyedMapState;
import org.apache.flink.util.Preconditions;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * An implementation of {@link KeyedMapState} backed by a state storage.
 *
 * @param <K> Type of the keys in the state.
 * @param <MK> Type of the map keys in the state.
 * @param <MV> Type of theGemini map values in the state.
 */
public abstract class AbstractGeminiKeyedMapStateImpl<K, MK, MV, M extends Map<MK, MV>>
	implements AbstractKeyedMapState<K, MK, MV, M> {

	private final AbstractGTableKeyedMapImpl<K, MK, MV> table;

	/**
	 * Constructor with the state storage to store mappings.
	 */
	public AbstractGeminiKeyedMapStateImpl(
		AbstractGTableKeyedMapImpl<K, MK, MV> table
	) {
		this.table = Preconditions.checkNotNull(table);
	}

	//--------------------------------------------------------------------------

	@Override
	public boolean contains(K key) {
		if (key == null) {
			return false;
		}

		return getRegion(key).contains(key);
	}

	@Override
	public boolean contains(K key, MK mapKey) {
		if (key == null || mapKey == null) {
			return false;
		}

		return getRegion(key).contains(key, mapKey);
	}

	@Override
	public M get(K key) {
		return getOrDefault(key, null);
	}

	@SuppressWarnings("unchecked")
	@Override
	public M getOrDefault(K key, M defaultValue) {
		if (key == null) {
			return defaultValue;
		}

		Map<MK, MV> result = getRegion(key).getOrDefault(key, null);
		return result == null ? defaultValue : (M) result;
	}

	@Override
	public MV get(K key, MK mapKey) {
		return getOrDefault(key, mapKey, null);
	}

	@SuppressWarnings("unchecked")
	@Override
	public MV getOrDefault(K key, MK mapKey, MV defaultMapValue) {
		if (key == null || mapKey == null) {
			return defaultMapValue;
		}

		return getRegion(key).getOrDefault(key, mapKey, defaultMapValue);
	}

	@Override
	public Map<K, M> getAll(Collection<? extends K> keys) {
		if (keys == null || keys.isEmpty()) {
			return new HashMap<>();
		}

		Map<K, M> results = new HashMap<>();
		for (K key : keys) {
			if (key == null) {
				continue;
			}

			M result = get(key);
			if (result != null && !result.isEmpty()) {
				results.put(key, result);
			}
		}

		return results;
	}

	@SuppressWarnings("unchecked")
	@Override
	public M getAll(K key, Collection<? extends MK> mapKeys) {
		if (key == null || mapKeys == null || mapKeys.isEmpty()) {
			return createMap();
		}

		// TODO performance
		M results = createMap();

		for (MK mapKey : mapKeys) {
			if (mapKey != null) {
				MV value = get(key, mapKey);
				if (value != null) {
					results.put(mapKey, value);
				}
			}
		}
		return results;
	}

	@SuppressWarnings("unchecked")
	@Override
	public Map<K, M> getAll(Map<K, ? extends Collection<? extends MK>> map) {
		if (map == null || map.isEmpty()) {
			return Collections.emptyMap();
		}

		Map<K, M> results = new HashMap<>();

		for (Map.Entry<K, ? extends Collection<? extends MK>> entry : map.entrySet()) {
			K key = entry.getKey();
			Collection<? extends MK> mapKeys = entry.getValue();
			M resultMap = getAll(key, mapKeys);
			if (!resultMap.isEmpty()) {
				results.put(key, getAll(key, mapKeys));
			}
		}
		return results;
	}

	@Override
	public void add(K key, MK mapKey, MV mapValue) {
		Preconditions.checkNotNull(key);

		getRegion(key).add(key, mapKey, mapValue);
	}

	@Override
	public void addAll(K key, Map<? extends MK, ? extends MV> mappings) {
		Preconditions.checkNotNull(key);

		if (mappings == null || mappings.isEmpty()) {
			return;
		}

		getRegion(key).addAll(key, mappings);
	}

	@Override
	public void addAll(Map<? extends K, ? extends Map<? extends MK, ? extends MV>> map) {
		if (map == null || map.isEmpty()) {
			return;
		}

		for (Map.Entry entry : map.entrySet()) {
			addAll((K) entry.getKey(), (Map) entry.getValue());
		}
	}

	@Override
	public void remove(K key) {
		if (key == null) {
			return;
		}

		getRegion(key).remove(key);
	}

	@Override
	public void remove(K key, MK mapKey) {
		if (key == null) {
			return;
		}

		getRegion(key).remove(key, mapKey);
	}

	@Override
	public void removeAll(Collection<? extends K> keys) {
		if (keys == null || keys.isEmpty()) {
			return;
		}

		for (K key : keys) {
			remove(key);
		}
	}

	@Override
	public void removeAll(K key, Collection<? extends MK> mapKeys) {
		if (key == null || mapKeys == null || mapKeys.isEmpty()) {
			return;
		}

		// TODO performance
		for (MK mapKey : mapKeys) {
			if (mapKey != null) {
				remove(key, mapKey);
			}
		}
	}

	@Override
	public void removeAll(Map<? extends K, ? extends Collection<? extends MK>> map) {
		if (map == null || map.isEmpty()) {
			return;
		}

		for (Map.Entry<? extends K, ? extends Collection<? extends MK>> entry : map.entrySet()) {
			removeAll(entry.getKey(), entry.getValue());
		}
	}

	@Override
	public Iterator<Map.Entry<MK, MV>> iterator(K key) {
		Preconditions.checkNotNull(key);

		// TODO performance
		Map<MK, MV> res = getRegion(key).get(key);
		if (res == null) {
			return Collections.emptyIterator();
		} else {
			return res.entrySet().iterator();
		}
	}

	@Override
	public Iterable<Map.Entry<MK, MV>> entries(K key) {
		Preconditions.checkNotNull(key);

		// TODO performance
		Map<MK, MV> res = getRegion(key).get(key);
		if (res == null) {
			return Collections.emptySet();
		} else {
			return res.entrySet();
		}
	}

	@Override
	public Iterable<MK> mapKeys(K key) {
		Preconditions.checkNotNull(key);

		// TODO performance
		Map<MK, MV> res = getRegion(key).get(key);
		if (res == null) {
			return Collections.emptySet();
		} else {
			return res.keySet();
		}
	}

	@Override
	public Iterable<MV> mapValues(K key) {
		Preconditions.checkNotNull(key);

		// TODO performance
		Map<MK, MV> res = getRegion(key).get(key);
		if (res == null) {
			return Collections.emptySet();
		} else {
			return res.values();
		}
	}

	@Override
	public Map<K, M> getAll() {
		Map<K, Map<MK, MV>> results = new HashMap<>();
		Iterator<GRegion> iterator = table.regionIterator();
		while (iterator.hasNext()) {
			GRegionKMapImpl<K, MK, MV> region = (GRegionKMapImpl<K, MK, MV>) iterator.next();
			region.getAll(results);
		}
		return (Map<K, M>) results;
	}

	@Override
	public void removeAll() {
		Iterator<GRegion> iterator = table.regionIterator();
		while (iterator.hasNext()) {
			GRegionKMapImpl<K, MK, MV> region = (GRegionKMapImpl<K, MK, MV>) iterator.next();
			region.removeAll();
		}
	}

	@Override
	public Iterable<K> keys() {
		// TODO complexity
		return getAll().keySet();
	}

	@Override
	public StateStorage<K, M> getStateStorage() {
		throw new UnsupportedOperationException();
	}

	@SuppressWarnings("unchecked")
	protected AbstractGRegionKMapImpl<K, MK, MV, ? extends Map<MK, MV>> getRegion(K key) {
		return (AbstractGRegionKMapImpl<K, MK, MV, ? extends Map<MK, MV>>) table.getRegion(key);
	}

	abstract M createMap();
}
