/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.keyed;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.core.memory.DataOutputViewStreamWrapper;
import org.apache.flink.queryablestate.client.state.serialization.KvStateSerializer;
import org.apache.flink.runtime.state.StateStorage;
import org.apache.flink.runtime.state.VoidNamespaceSerializer;
import org.apache.flink.runtime.state.gemini.engine.GRegion;
import org.apache.flink.runtime.state.gemini.engine.hashtable.GRegionKListImpl;
import org.apache.flink.runtime.state.gemini.engine.hashtable.GTableKeyedListImpl;
import org.apache.flink.runtime.state.keyed.KeyedListState;
import org.apache.flink.runtime.state.keyed.KeyedListStateDescriptor;
import org.apache.flink.util.Preconditions;

import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * An implementation of {@link KeyedListState} backed by a {@link StateStorage}.
 * The pairs are formatted as {K -> List{E}}, and the pairs are partitioned by K.
 *
 * @param <K> Type of the keys in the state.
 * @param <E> Type of the elements in the state.
 */
public final class GeminiKeyedListStateImpl<K, E> implements KeyedListState<K, E> {

	/**
	 * The descriptor of current state.
	 */
	private final KeyedListStateDescriptor<K, E> descriptor;

	private final GTableKeyedListImpl<K, E> table;

	//--------------------------------------------------------------------------

	/**
	 * Constructor with the state storage to store the values.
	 *
	 * @param descriptor The descriptor of this state.
	 */
	public GeminiKeyedListStateImpl(
		KeyedListStateDescriptor<K, E> descriptor,
		GTableKeyedListImpl<K, E> table) {
		this.descriptor = Preconditions.checkNotNull(descriptor);
		this.table = Preconditions.checkNotNull(table);
	}

	@Override
	public KeyedListStateDescriptor getDescriptor() {
		return descriptor;
	}

	//--------------------------------------------------------------------------

	@Override
	public boolean contains(K key) {
		List<E> list = getOrDefault(key, null);

		return list != null;
	}

	@SuppressWarnings("unchecked")
	@Override
	public List<E> get(K key) {
		return getOrDefault(key, null);
	}

	@Override
	public List<E> getOrDefault(K key, List<E> defaultValue) {
		if (key == null) {
			return defaultValue;
		}

		List<E> result = getRegion(key).get(key);
		return result == null ? defaultValue : result;
	}

	@Override
	public Map<K, List<E>> getAll(Collection<? extends K> keys) {
		if (keys == null || keys.isEmpty()) {
			return Collections.emptyMap();
		}

		Map<K, List<E>> results = new HashMap<>();

		for (K key : keys) {
			if (key == null) {
				continue;
			}

			List<E> result = get(key);
			if (result != null && !result.isEmpty()) {
				results.put(key, result);
			}
		}

		return results;
	}

	@Override
	public void add(K key, E element) {
		Preconditions.checkNotNull(key);
		Preconditions.checkNotNull(element, "You can not add null value to list state.");

		getRegion(key).add(key, element);
	}

	@Override
	public void addAll(K key, Collection<? extends E> elements) {
		Preconditions.checkNotNull(key);
		Preconditions.checkNotNull(elements, "List of values to add cannot be null.");

		if (elements.isEmpty()) {
			return;
		}

		getRegion(key).addAll(key, elements);
	}

	@Override
	public void addAll(Map<? extends K, ? extends Collection<? extends E>> map) {
		if (map == null || map.isEmpty()) {
			return;
		}

		for (Map.Entry<? extends K, ? extends Collection<? extends E>> entry : map.entrySet()) {
			addAll(entry.getKey(), entry.getValue());
		}
	}

	@Override
	public void put(K key, E element) {
		Preconditions.checkNotNull(key);
		Preconditions.checkNotNull(element, "You can not add null value to list state.");

		getRegion(key).put(key, new ArrayList<>(Arrays.asList(element)));
	}

	@Override
	public void putAll(K key, Collection<? extends E> elements) {
		Preconditions.checkNotNull(key);

		List<E> list = new ArrayList<>();
		for (E element : elements) {
			Preconditions.checkNotNull(element, "You cannot add null to a ListState.");
			list.add(element);
		}
		getRegion(key).put(key, list);

	}

	@Override
	public void putAll(Map<? extends K, ? extends Collection<? extends E>> map) {
		if (map == null || map.isEmpty()) {
			return;
		}

		for (Map.Entry<? extends K, ? extends Collection<? extends E>> entry : map.entrySet()) {
			putAll(entry.getKey(), entry.getValue());
		}
	}

	@Override
	public void remove(K key) {
		if (key == null) {
			return;
		}

		getRegion(key).remove(key);
	}

	@Override
	public boolean remove(K key, E elementToRemove) {
		if (key == null) {
			return false;
		}

		// TODO remove should return a value
		getRegion(key).remove(key, elementToRemove);

		return true;
	}

	@Override
	public void removeAll(Collection<? extends K> keys) {
		if (keys == null || keys.isEmpty()) {
			return;
		}

		for (K key : keys) {
			remove(key);
		}
	}

	@Override
	public boolean removeAll(K key, Collection<? extends E> elementsToRemove) {
		if (key == null) {
			return false;
		}

		// TODO removeAll should return a value
		getRegion(key).removeAll(key, elementsToRemove);

		return true;
	}

	@Override
	public boolean removeAll(Map<? extends K, ? extends Collection<? extends E>> map) {
		if (map == null || map.isEmpty()) {
			return false;
		}

		boolean success = false;
		for (Map.Entry<? extends K, ? extends Collection<? extends E>> entry : map.entrySet()) {
			K key = entry.getKey();
			Collection<? extends E> elements = entry.getValue();
			success = removeAll(key, elements) || success;
		}

		return success;
	}

	@Override
	public Map<K, List<E>> getAll() {
		Map<K, List<E>> results = new HashMap<>();
		Iterator<GRegion> iterator = table.regionIterator();
		while (iterator.hasNext()) {
			GRegionKListImpl<K, E> region = (GRegionKListImpl<K, E>) iterator.next();
			region.getAll(results);
		}
		return results;
	}

	@Override
	public void removeAll() {
		Iterator<GRegion> iterator = table.regionIterator();
		while (iterator.hasNext()) {
			GRegionKListImpl<K, List<E>> region = (GRegionKListImpl<K, List<E>>) iterator.next();
			region.removeAll();
		}
	}

	@Override
	public Iterable<K> keys() {
		// TODO complexity
		return getAll().keySet();
	}

	@Override
	public E poll(K key) {
		return getRegion(key).poll(key);
	}

	@Override
	public E peek(K key) {
		return getRegion(key).peek(key);
	}

	@Override
	public byte[] getSerializedValue(
		final byte[] serializedKeyAndNamespace,
		final TypeSerializer<K> safeKeySerializer,
		final TypeSerializer<List<E>> safeValueSerializer) throws Exception {

		K key = KvStateSerializer.deserializeKeyAndNamespace(serializedKeyAndNamespace,
			safeKeySerializer,
			VoidNamespaceSerializer.INSTANCE).f0;

		List<E> value = get(key);
		if (value == null) {
			return null;
		}

		final TypeSerializer<E> dupSerializer = descriptor.getElementSerializer();

		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		DataOutputViewStreamWrapper view = new DataOutputViewStreamWrapper(baos);

		for (int i = 0; i < value.size(); i++) {
			dupSerializer.serialize(value.get(i), view);
			if (i < value.size() - 1) {
				view.writeByte(',');
			}
		}
		view.flush();

		return baos.toByteArray();
	}

	@Override
	public StateStorage<K, List<E>> getStateStorage() {
		throw new UnsupportedOperationException();
	}

	@SuppressWarnings("unchecked")
	private GRegionKListImpl<K, E> getRegion(K key) {
		return (GRegionKListImpl<K, E>) table.getRegion(key);
	}
}
