/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.keyed;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.runtime.state.gemini.engine.hashtable.AbstractGTableKeyedMapImpl;
import org.apache.flink.runtime.state.gemini.engine.hashtable.GRegionKSortedMapImpl;
import org.apache.flink.runtime.state.keyed.KeyedSortedMapState;
import org.apache.flink.runtime.state.keyed.KeyedSortedMapStateDescriptor;
import org.apache.flink.runtime.state.keyed.KeyedStateDescriptor;

import java.util.Iterator;
import java.util.Map;
import java.util.SortedMap;
import java.util.TreeMap;

import static org.apache.flink.util.Preconditions.checkNotNull;

/**
 * An implementation of {@link KeyedSortedMapState} backed by a state storage.
 *
 * @param <K> Type of the keys in the state.
 * @param <MK> Type of the map keys in the state.
 * @param <MV> Type of the map values in the state.
 */
public final class GeminiKeyedSortedMapStateImpl<K, MK, MV>
	extends AbstractGeminiKeyedMapStateImpl<K, MK, MV, SortedMap<MK, MV>>
	implements KeyedSortedMapState<K, MK, MV> {

	/**
	 * The descriptor of current state.
	 */
	private KeyedSortedMapStateDescriptor<K, MK, MV> stateDescriptor;

	/**
	 * Constructor with the state storage to store mappings.
	 *
	 * @param descriptor The descriptor of current state.
	 */
	public GeminiKeyedSortedMapStateImpl(
		KeyedSortedMapStateDescriptor<K, MK, MV> descriptor,
		AbstractGTableKeyedMapImpl<K, MK, MV> table) {
		super(table);
		this.stateDescriptor = checkNotNull(descriptor);
	}

	@Override
	SortedMap<MK, MV> createMap() {
		return new TreeMap<>();
	}

	@Override
	public KeyedStateDescriptor getDescriptor() {
		return stateDescriptor;
	}

	@Override
	public byte[] getSerializedValue(byte[] serializedKeyAndNamespace, TypeSerializer<K> safeKeySerializer, TypeSerializer<SortedMap<MK, MV>> safeValueSerializer) throws Exception {
		return new byte[0];
	}

	@Override
	public Map.Entry<MK, MV> firstEntry(K key) {
		if (key == null) {
			return null;
		}

		GRegionKSortedMapImpl<K, MK, MV> region = (GRegionKSortedMapImpl<K, MK, MV>) getRegion(key);
		return region.firstEntry(key);
	}

	@Override
	public Map.Entry<MK, MV> lastEntry(K key) {
		if (key == null) {
			return null;
		}

		GRegionKSortedMapImpl<K, MK, MV> region = (GRegionKSortedMapImpl<K, MK, MV>) getRegion(key);
		return region.lastEntry(key);
	}

	@Override
	public Iterator<Map.Entry<MK, MV>> headIterator(K key, MK endMapKey) {
		if (key == null || endMapKey == null) {
			return null;
		}

		GRegionKSortedMapImpl<K, MK, MV> region = (GRegionKSortedMapImpl<K, MK, MV>) getRegion(key);
		return region.headIterator(key, endMapKey);
	}

	@Override
	public Iterator<Map.Entry<MK, MV>> tailIterator(K key, MK startMapKey) {
		if (key == null || startMapKey == null) {
			return null;
		}

		GRegionKSortedMapImpl<K, MK, MV> region = (GRegionKSortedMapImpl<K, MK, MV>) getRegion(key);
		return region.tailIterator(key, startMapKey);
	}

	@Override
	public Iterator<Map.Entry<MK, MV>> subIterator(K key, MK startMapKey, MK endMapKey) {
		if (key == null || startMapKey == null || endMapKey == null) {
			return null;
		}

		GRegionKSortedMapImpl<K, MK, MV> region = (GRegionKSortedMapImpl<K, MK, MV>) getRegion(key);
		return region.subIterator(key, startMapKey, endMapKey);
	}
}

