/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.keyed;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.queryablestate.client.state.serialization.KvStateSerializer;
import org.apache.flink.runtime.state.StateAccessException;
import org.apache.flink.runtime.state.StateStorage;
import org.apache.flink.runtime.state.StateTransformationFunction;
import org.apache.flink.runtime.state.VoidNamespaceSerializer;
import org.apache.flink.runtime.state.gemini.engine.GRegion;
import org.apache.flink.runtime.state.gemini.engine.hashtable.GRegionKVImpl;
import org.apache.flink.runtime.state.gemini.engine.hashtable.GTableOneKeyImpl;
import org.apache.flink.runtime.state.keyed.KeyedValueState;
import org.apache.flink.runtime.state.keyed.KeyedValueStateDescriptor;
import org.apache.flink.util.Preconditions;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 * An implementation of {@link KeyedValueState} based on a {@link StateStorage}
 * The pairs are formatted as {K -> V}, and are partitioned by K.
 *
 * @param <K> Type of the keys in the state.
 * @param <V> Type of the values in the state.
 */
public final class GeminiKeyedValueStateImpl<K, V> implements KeyedValueState<K, V> {

	/**
	 * The descriptor of this state.
	 */
	private final KeyedValueStateDescriptor<K, V> descriptor;

	private final GTableOneKeyImpl<K, V> table;

	/**
	 * Constructor with the state storage to store the values.
	 *
	 * @param descriptor The descriptor of this state.
	 */
	public GeminiKeyedValueStateImpl(
		KeyedValueStateDescriptor<K, V> descriptor,
		GTableOneKeyImpl<K, V> table
	) {
		this.descriptor = Preconditions.checkNotNull(descriptor);
		this.table = Preconditions.checkNotNull(table);
	}

	@Override
	public KeyedValueStateDescriptor getDescriptor() {
		return descriptor;
	}

	//--------------------------------------------------------------------------

	@Override
	public boolean contains(K key) {
		return get(key) != null;
	}

	@Override
	public V get(K key) {
		return getOrDefault(key, null);
	}

	@Override
	public V getOrDefault(K key, V defaultValue) {
		if (key == null) {
			return defaultValue;
		}

		V value = getRegion(key).get(key);

		return value == null ? defaultValue : value;
	}

	@Override
	public Map<K, V> getAll(Collection<? extends K> keys) {
		if (keys == null || keys.isEmpty()) {
			return Collections.emptyMap();
		}

		Map<K, V> results = new HashMap<>();

		for (K key : keys) {
			if (key == null) {
				continue;
			}
			V value = get(key);
			if (value != null) {
				results.put(key, value);
			}
		}

		return results;
	}

	@Override
	public void remove(K key) {
		if (key == null) {
			return;
		}

		getRegion(key).remove(key);
	}

	@Override
	public void removeAll(Collection<? extends K> keys) {
		if (keys == null || keys.isEmpty()) {
			return;
		}

		for (K key : keys) {
			remove(key);
		}
	}

	@Override
	public void put(K key, V value) {
		Preconditions.checkNotNull(key);

		getRegion(key).put(key, value);
	}

	@Override
	public void putAll(Map<? extends K, ? extends V> pairs) {
		if (pairs == null || pairs.isEmpty()) {
			return;
		}

		for (Map.Entry<? extends K, ? extends V> entry : pairs.entrySet()) {
			put(entry.getKey(), entry.getValue());
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public Map<K, V> getAll() {
		Map<K, V> results = new HashMap<>();
		Iterator<GRegion> iterator = table.regionIterator();
		while (iterator.hasNext()) {
			GRegionKVImpl<K, V> region = (GRegionKVImpl<K, V>) iterator.next();
			region.getAll(results);
		}
		return results;
	}

	@Override
	public void removeAll() {
		Iterator<GRegion> iterator = table.regionIterator();
		while (iterator.hasNext()) {
			GRegionKVImpl<K, V> region = (GRegionKVImpl<K, V>) iterator.next();
			region.removeAll();
		}
	}

	@Override
	public Iterable<K> keys() {
		// TODO complexity
		return getAll().keySet();
	}

	@Override
	public <T> void transform(K key, T value, StateTransformationFunction<V, T> transformation) {
		try {
			// TODO optimize the implementation
			V oldValue = get(key);
			V newValue = transformation.apply(oldValue, value);
			put(key, newValue);
		} catch (Exception e) {
			throw new StateAccessException(e);
		}
	}

	@Override
	public byte[] getSerializedValue(
		final byte[] serializedKeyAndNamespace,
		final TypeSerializer<K> safeKeySerializer,
		final TypeSerializer<V> safeValueSerializer) throws Exception {
		K key = KvStateSerializer.deserializeKeyAndNamespace(serializedKeyAndNamespace, safeKeySerializer, VoidNamespaceSerializer.INSTANCE).f0;

		V value = get(key);
		if (value == null) {
			return null;
		}

		return KvStateSerializer.serializeValue(value, descriptor.getValueSerializer());
	}

	@Override
	public StateStorage<K, V> getStateStorage() {
		throw new UnsupportedOperationException();
	}

	@SuppressWarnings("unchecked")
	private GRegionKVImpl<K, V> getRegion(K key) {
		return table.getRegion(key);
	}
}
