/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.runtime.state.gemini.subkeyed;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.runtime.state.StateAccessException;
import org.apache.flink.runtime.state.StateStorage;
import org.apache.flink.runtime.state.StateTransformationFunction;
import org.apache.flink.runtime.state.gemini.engine.GRegion;
import org.apache.flink.runtime.state.gemini.engine.hashtable.GRegionKMapImpl;
import org.apache.flink.runtime.state.gemini.engine.hashtable.GTableSubKeyedValueImpl;
import org.apache.flink.runtime.state.subkeyed.SubKeyedValueState;
import org.apache.flink.runtime.state.subkeyed.SubKeyedValueStateDescriptor;

import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static org.apache.flink.util.Preconditions.checkNotNull;

/**
 * An implementation of {@link SubKeyedValueState} based on an {@link StateStorage}
 * The pairs in the state storage are formatted as {(K, N) -> V}, and are
 * partitioned by K.
 *
 * @param <K> Type of the keys in the state.
 * @param <N> Type of the namespaces in the state.
 * @param <V> Type of the values in the state.
 */
public final class GeminiSubKeyedValueStateImpl<K, N, V> implements SubKeyedValueState<K, N, V> {

	/**
	 * The descriptor of this state.
	 */
	private final SubKeyedValueStateDescriptor descriptor;

	private final GTableSubKeyedValueImpl<K, N, V> table;
	/**
	 * Constructor with the state storage to store the values.
	 *
	 * @param descriptor The descriptor of this state.
	 * @param table
	 */
	public GeminiSubKeyedValueStateImpl(
		SubKeyedValueStateDescriptor descriptor,
		GTableSubKeyedValueImpl<K, N, V> table
	) {
		this.descriptor = checkNotNull(descriptor);
		this.table = checkNotNull(table);
	}

	@Override
	public SubKeyedValueStateDescriptor getDescriptor() {
		return descriptor;
	}

	//--------------------------------------------------------------------------

	@Override
	public boolean contains(K key, N namespace) {
		if (key == null || namespace == null) {
			return false;
		}

		return getRegion(key).contains(key, namespace);
	}

	@Override
	public V get(K key, N namespace) {
		return getOrDefault(key, namespace, null);
	}

	@SuppressWarnings("unchecked")
	@Override
	public V getOrDefault(K key, N namespace, V defaultValue) {
		if (key == null || namespace == null) {
			return defaultValue;
		}

		return getRegion(key).getOrDefault(key, namespace, defaultValue);
	}

	@SuppressWarnings("unchecked")
	@Override
	public Map<N, V> getAll(K key) {
		if (key == null) {
			return Collections.emptyMap();
		}

		Map<N, V> result = getRegion(key).get(key);
		return result == null ? Collections.emptyMap() : result;
	}

	@Override
	public void remove(K key, N namespace) {
		if (key == null || namespace == null) {
			return;
		}

		getRegion(key).remove(key, namespace);
	}

	@Override
	public void removeAll(K key) {
		if (key == null) {
			return;
		}

		getRegion(key).remove(key);
	}

	@Override
	public void put(K key, N namespace, V value) {
		checkNotNull(key);
		checkNotNull(namespace);

		getRegion(key).add(key, namespace, value);
	}

	@Override
	public V getAndRemove(K key, N namespace) {
		if (key == null || namespace == null) {
			return null;
		}

		V value = get(key, namespace);
		remove(key, namespace);
		return value;
	}

	@Override
	public Iterator<N> iterator(K key) {
		checkNotNull(key);

		Map<N, V> ret = getRegion(key).get(key);
		if (ret == null) {
			return Collections.emptyIterator();
		} else {
			Iterator<N> namespaceIter = ret.keySet().iterator();
			return new Iterator<N>() {
				N currentNamespace = null;

				@Override
				public boolean hasNext() {
					return namespaceIter.hasNext();
				}

				@Override
				public N next() {
					currentNamespace = namespaceIter.next();
					return currentNamespace;
				}

				@Override
				public void remove() {
					GeminiSubKeyedValueStateImpl.this.remove(key, currentNamespace);
				}
			};
		}
	}

	@Override
	public StateStorage<K, V> getStateStorage() {
		throw new UnsupportedOperationException();
	}

	@Override
	public <T> void transform(K key, N namespace, T value, StateTransformationFunction<V, T> transformation) {
		V oldValue = get(key, namespace);
		try {
			V newValue = transformation.apply(oldValue, value);
			put(key, namespace, newValue);
		} catch (Exception e) {
			throw new StateAccessException(e);
		}
	}

	@Override
	public Iterable<K> keys(N namespace) {
		Iterator<GRegion> iterator = table.regionIterator();
		Set<K> result = new HashSet<>();
		while (iterator.hasNext()) {
			GRegionKMapImpl<K, N, V> cur = (GRegionKMapImpl<K, N, V>) iterator.next();
			result.addAll(
				cur.getAll().entrySet().stream()
					.filter(a -> a.getValue().keySet().contains(namespace))
					.map(a -> a.getKey())
					.collect(Collectors.toSet()));
		}
		return result.size() == 0 ? null : result;
	}

	@Override
	public byte[] getSerializedValue(
		byte[] serializedKeyAndNamespace,
		TypeSerializer<K> safeKeySerializer,
		TypeSerializer<N> safeNamespaceSerializer,
		TypeSerializer<V> safeValueSerializer) throws Exception {

		// TODO
		throw new UnsupportedOperationException();
	}

	@SuppressWarnings("unchecked")
	private GRegionKMapImpl<K, N, V> getRegion(K key) {
		return table.getRegion(key);
	}
}
