/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.taskexecutor.slot;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.UUID;
import javax.annotation.Nullable;
import org.apache.flink.api.common.JobID;
import org.apache.flink.api.common.time.Time;
import org.apache.flink.runtime.clusterframework.types.AllocationID;
import org.apache.flink.runtime.clusterframework.types.ResourceID;
import org.apache.flink.runtime.clusterframework.types.ResourceProfile;
import org.apache.flink.runtime.clusterframework.types.SlotID;
import org.apache.flink.runtime.executiongraph.ExecutionAttemptID;
import org.apache.flink.runtime.resourcemanager.placementconstraint.SlotTag;
import org.apache.flink.runtime.taskexecutor.SlotReport;
import org.apache.flink.runtime.taskexecutor.SlotStatus;
import org.apache.flink.runtime.taskexecutor.slot.SlotActions;
import org.apache.flink.runtime.taskexecutor.slot.SlotNotActiveException;
import org.apache.flink.runtime.taskexecutor.slot.SlotNotFoundException;
import org.apache.flink.runtime.taskexecutor.slot.TaskSlot;
import org.apache.flink.runtime.taskexecutor.slot.TaskSlotState;
import org.apache.flink.runtime.taskexecutor.slot.TimeoutListener;
import org.apache.flink.runtime.taskexecutor.slot.TimerService;
import org.apache.flink.runtime.taskmanager.Task;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TaskSlotTable
implements TimeoutListener<AllocationID> {
    private static final Logger LOG = LoggerFactory.getLogger(TaskSlotTable.class);
    private final TimerService<AllocationID> timerService;
    private final ResourceProfile totalResource;
    private final List<TaskSlot> taskSlots;
    private final Map<AllocationID, TaskSlot> allocationIDTaskSlotMap;
    private final Map<ExecutionAttemptID, TaskSlotMapping> taskSlotMappings;
    private final Map<JobID, Set<AllocationID>> slotsPerJob;
    private SlotActions slotActions;
    private boolean started;

    public TaskSlotTable(Collection<ResourceProfile> resourceProfiles, ResourceProfile totalResource, TimerService<AllocationID> timerService) {
        int numberSlots = resourceProfiles.size();
        Preconditions.checkArgument((0 < numberSlots ? 1 : 0) != 0, (Object)"The number of task slots must be greater than 0.");
        this.timerService = (TimerService)Preconditions.checkNotNull(timerService);
        this.taskSlots = Arrays.asList(new TaskSlot[numberSlots]);
        int index = 0;
        for (ResourceProfile resourceProfile : resourceProfiles) {
            this.taskSlots.set(index, new TaskSlot(index, resourceProfile));
            ++index;
        }
        this.totalResource = totalResource;
        this.allocationIDTaskSlotMap = new HashMap<AllocationID, TaskSlot>(numberSlots);
        this.taskSlotMappings = new HashMap<ExecutionAttemptID, TaskSlotMapping>(4 * numberSlots);
        this.slotsPerJob = new HashMap<JobID, Set<AllocationID>>(4);
        this.slotActions = null;
        this.started = false;
    }

    public void start(SlotActions initialSlotActions) {
        this.slotActions = (SlotActions)Preconditions.checkNotNull((Object)initialSlotActions);
        this.timerService.start(this);
        this.started = true;
    }

    public void stop() {
        this.started = false;
        this.timerService.stop();
        this.slotActions = null;
    }

    public Set<AllocationID> getAllocationIdsPerJob(JobID jobId) {
        Set<AllocationID> allocationIds = this.slotsPerJob.get(jobId);
        if (allocationIds == null) {
            return Collections.emptySet();
        }
        return Collections.unmodifiableSet(allocationIds);
    }

    public SlotReport createSlotReport(ResourceID resourceId) {
        int numberSlots = this.taskSlots.size();
        List<SlotStatus> slotStatuses = Arrays.asList(new SlotStatus[numberSlots]);
        for (int i = 0; i < numberSlots; ++i) {
            TaskSlot taskSlot = this.taskSlots.get(i);
            SlotID slotId = new SlotID(resourceId, taskSlot.getIndex());
            SlotStatus slotStatus = new SlotStatus(slotId, taskSlot.getResourceProfile(), taskSlot.getJobId(), taskSlot.getAllocationId(), taskSlot.getAllocationResourceProfile(), taskSlot.getSlotTags(), taskSlot.getVersion());
            slotStatuses.set(i, slotStatus);
        }
        SlotReport slotReport = new SlotReport(slotStatuses);
        return slotReport;
    }

    public boolean allocateSlot(int index, JobID jobId, AllocationID allocationId, ResourceProfile allocationResourceProfile, List<SlotTag> tags, Time slotTimeout) {
        this.checkInit();
        TaskSlot taskSlot = this.taskSlots.get(index);
        if (!this.hasEnoughResource(allocationResourceProfile)) {
            LOG.warn("Not enough resource for allocation with job = {},  id = {}, resource = {}", new Object[]{jobId, allocationId, allocationResourceProfile});
            return false;
        }
        boolean result = taskSlot.allocate(jobId, allocationId, allocationResourceProfile, tags);
        if (result) {
            this.allocationIDTaskSlotMap.put(allocationId, taskSlot);
            this.timerService.registerTimeout(allocationId, slotTimeout.getSize(), slotTimeout.getUnit());
            Set<AllocationID> slots = this.slotsPerJob.get(jobId);
            if (slots == null) {
                slots = new HashSet<AllocationID>(4);
                this.slotsPerJob.put(jobId, slots);
            }
            slots.add(allocationId);
        }
        return result;
    }

    public boolean markSlotActive(AllocationID allocationId) throws SlotNotFoundException {
        this.checkInit();
        TaskSlot taskSlot = this.getTaskSlot(allocationId);
        if (taskSlot != null) {
            if (taskSlot.markActive()) {
                LOG.info("Activate slot {}.", (Object)allocationId);
                this.timerService.unregisterTimeout(allocationId);
                return true;
            }
            return false;
        }
        throw new SlotNotFoundException(allocationId);
    }

    public boolean markSlotInactive(AllocationID allocationId, Time slotTimeout) throws SlotNotFoundException {
        this.checkInit();
        TaskSlot taskSlot = this.getTaskSlot(allocationId);
        if (taskSlot != null) {
            if (taskSlot.markInactive()) {
                this.timerService.registerTimeout(allocationId, slotTimeout.getSize(), slotTimeout.getUnit());
                return true;
            }
            return false;
        }
        throw new SlotNotFoundException(allocationId);
    }

    public int freeSlot(AllocationID allocationId) throws SlotNotFoundException {
        return this.freeSlot(allocationId, new Exception("The task slot of this task is being freed."));
    }

    public int freeSlot(AllocationID allocationId, Throwable cause) throws SlotNotFoundException {
        this.checkInit();
        TaskSlot taskSlot = this.getTaskSlot(allocationId);
        if (taskSlot != null) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Free slot {}.", (Object)taskSlot, (Object)cause);
            } else {
                LOG.info("Free slot {}.", (Object)taskSlot);
            }
            JobID jobId = taskSlot.getJobId();
            if (taskSlot.markFree()) {
                this.allocationIDTaskSlotMap.remove((Object)allocationId);
                this.timerService.unregisterTimeout(allocationId);
                Set<AllocationID> slots = this.slotsPerJob.get(jobId);
                if (slots == null) {
                    throw new IllegalStateException("There are no more slots allocated for the job " + jobId + ". This indicates a programming bug.");
                }
                slots.remove((Object)allocationId);
                if (slots.isEmpty()) {
                    this.slotsPerJob.remove(jobId);
                }
                return taskSlot.getIndex();
            }
            taskSlot.markReleasing();
            Iterator<Task> taskIterator = taskSlot.getTasks();
            while (taskIterator.hasNext()) {
                taskIterator.next().failExternally(cause);
            }
            return -1;
        }
        throw new SlotNotFoundException(allocationId);
    }

    public boolean isValidTimeout(AllocationID allocationId, UUID ticket) {
        this.checkInit();
        return this.timerService.isValid(allocationId, ticket);
    }

    public boolean isAllocated(int index, JobID jobId, AllocationID allocationId) {
        TaskSlot taskSlot = this.taskSlots.get(index);
        return taskSlot.isAllocated(jobId, allocationId);
    }

    public boolean existsActiveSlot(JobID jobId, AllocationID allocationId) {
        TaskSlot taskSlot = this.getTaskSlot(allocationId);
        if (taskSlot != null) {
            return taskSlot.isActive(jobId, allocationId);
        }
        return false;
    }

    public boolean isSlotFree(int index) {
        TaskSlot taskSlot = this.taskSlots.get(index);
        return taskSlot.isFree();
    }

    public boolean hasAllocatedSlots(JobID jobId) {
        return this.getAllocatedSlots(jobId).hasNext();
    }

    public Iterator<TaskSlot> getAllocatedSlots(JobID jobId) {
        return new TaskSlotIterator(jobId, TaskSlotState.ALLOCATED);
    }

    public Iterator<TaskSlot> getActiveSlots(JobID jobId) {
        return new TaskSlotIterator(jobId, TaskSlotState.ACTIVE);
    }

    @Nullable
    public JobID getOwningJob(AllocationID allocationId) {
        TaskSlot taskSlot = this.getTaskSlot(allocationId);
        if (taskSlot != null) {
            return taskSlot.getJobId();
        }
        return null;
    }

    public long getSlotVersion(int index) {
        return this.taskSlots.get(index).getVersion();
    }

    public void updateSlotVersion(int index, long version) {
        this.taskSlots.get(index).updateVersion(version);
    }

    public boolean addTask(Task task) throws SlotNotFoundException, SlotNotActiveException {
        Preconditions.checkNotNull((Object)task);
        TaskSlot taskSlot = this.getTaskSlot(task.getAllocationId());
        if (taskSlot != null) {
            if (taskSlot.isActive(task.getJobID(), task.getAllocationId())) {
                if (taskSlot.add(task)) {
                    this.taskSlotMappings.put(task.getExecutionId(), new TaskSlotMapping(task, taskSlot));
                    return true;
                }
                return false;
            }
            throw new SlotNotActiveException(task.getJobID(), task.getAllocationId());
        }
        throw new SlotNotFoundException(task.getAllocationId());
    }

    public Task removeTask(ExecutionAttemptID executionAttemptID) {
        this.checkInit();
        TaskSlotMapping taskSlotMapping = this.taskSlotMappings.remove((Object)executionAttemptID);
        if (taskSlotMapping != null) {
            Task task = taskSlotMapping.getTask();
            TaskSlot taskSlot = taskSlotMapping.getTaskSlot();
            taskSlot.remove(task.getExecutionId());
            if (taskSlot.isReleasing() && taskSlot.isEmpty()) {
                this.slotActions.freeSlot(taskSlot.getAllocationId());
            }
            return task;
        }
        return null;
    }

    public Task getTask(ExecutionAttemptID executionAttemptID) {
        TaskSlotMapping taskSlotMapping = this.taskSlotMappings.get((Object)executionAttemptID);
        if (taskSlotMapping != null) {
            return taskSlotMapping.getTask();
        }
        return null;
    }

    public Iterator<Task> getTasks(JobID jobId) {
        return new TaskIterator(jobId);
    }

    public AllocationID getCurrentAllocation(int index) {
        return this.taskSlots.get(index).getAllocationId();
    }

    @Override
    public void notifyTimeout(AllocationID key, UUID ticket) {
        this.checkInit();
        if (this.slotActions != null) {
            this.slotActions.timeoutSlot(key, ticket);
        }
    }

    @Nullable
    private TaskSlot getTaskSlot(AllocationID allocationId) {
        Preconditions.checkNotNull((Object)((Object)allocationId));
        return this.allocationIDTaskSlotMap.get((Object)allocationId);
    }

    private void checkInit() {
        Preconditions.checkState((boolean)this.started, (String)"The %s has to be started.", (Object[])new Object[]{TaskSlotTable.class.getSimpleName()});
    }

    private boolean hasEnoughResource(ResourceProfile allocationResourceProfile) {
        if (this.totalResource.equals(ResourceProfile.UNKNOWN)) {
            return true;
        }
        ResourceProfile remainResource = this.totalResource;
        for (TaskSlot taskSlot : this.taskSlots) {
            if (taskSlot.getAllocationId() == null) continue;
            remainResource = remainResource.minus(taskSlot.getAllocationResourceProfile());
        }
        return remainResource.isMatching(allocationResourceProfile);
    }

    private final class TaskIterator
    implements Iterator<Task> {
        private final Iterator<TaskSlot> taskSlotIterator;
        private Iterator<Task> currentTasks;

        private TaskIterator(JobID jobId) {
            this.taskSlotIterator = new TaskSlotIterator(jobId, TaskSlotState.ACTIVE);
            this.currentTasks = null;
        }

        @Override
        public boolean hasNext() {
            while ((this.currentTasks == null || !this.currentTasks.hasNext()) && this.taskSlotIterator.hasNext()) {
                TaskSlot taskSlot = this.taskSlotIterator.next();
                this.currentTasks = taskSlot.getTasks();
            }
            return this.currentTasks != null && this.currentTasks.hasNext();
        }

        @Override
        public Task next() {
            while (this.currentTasks == null || !this.currentTasks.hasNext()) {
                TaskSlot taskSlot;
                try {
                    taskSlot = this.taskSlotIterator.next();
                }
                catch (NoSuchElementException e) {
                    throw new NoSuchElementException("No more tasks.");
                }
                this.currentTasks = taskSlot.getTasks();
            }
            return this.currentTasks.next();
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("Cannot remove tasks via this iterator.");
        }
    }

    private final class TaskSlotIterator
    implements Iterator<TaskSlot> {
        private final Iterator<AllocationID> allSlots;
        private final TaskSlotState state;
        private TaskSlot currentSlot;

        private TaskSlotIterator(JobID jobId, TaskSlotState state) {
            Set allocationIds = (Set)TaskSlotTable.this.slotsPerJob.get(jobId);
            this.allSlots = allocationIds == null || allocationIds.isEmpty() ? Collections.emptyIterator() : allocationIds.iterator();
            this.state = (TaskSlotState)((Object)Preconditions.checkNotNull((Object)((Object)state)));
            this.currentSlot = null;
        }

        @Override
        public boolean hasNext() {
            while (this.currentSlot == null && this.allSlots.hasNext()) {
                AllocationID tempSlot = this.allSlots.next();
                TaskSlot taskSlot = TaskSlotTable.this.getTaskSlot(tempSlot);
                if (taskSlot == null || taskSlot.getState() != this.state) continue;
                this.currentSlot = taskSlot;
            }
            return this.currentSlot != null;
        }

        @Override
        public TaskSlot next() {
            AllocationID tempSlot;
            TaskSlot taskSlot;
            if (this.currentSlot != null) {
                TaskSlot result = this.currentSlot;
                this.currentSlot = null;
                return result;
            }
            do {
                try {
                    tempSlot = this.allSlots.next();
                }
                catch (NoSuchElementException e) {
                    throw new NoSuchElementException("No more task slots.");
                }
            } while ((taskSlot = TaskSlotTable.this.getTaskSlot(tempSlot)) == null || taskSlot.getState() != this.state);
            return taskSlot;
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("Cannot remove task slots via this iterator.");
        }
    }

    private final class AllocationIDIterator
    implements Iterator<AllocationID> {
        private final Iterator<TaskSlot> iterator;

        private AllocationIDIterator(JobID jobId, TaskSlotState state) {
            this.iterator = new TaskSlotIterator(jobId, state);
        }

        @Override
        public boolean hasNext() {
            return this.iterator.hasNext();
        }

        @Override
        public AllocationID next() {
            try {
                return this.iterator.next().getAllocationId();
            }
            catch (NoSuchElementException e) {
                throw new NoSuchElementException("No more allocation ids.");
            }
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("Cannot remove allocation ids via this iterator.");
        }
    }

    private static final class TaskSlotMapping {
        private final Task task;
        private final TaskSlot taskSlot;

        private TaskSlotMapping(Task task, TaskSlot taskSlot) {
            this.task = (Task)Preconditions.checkNotNull((Object)task);
            this.taskSlot = (TaskSlot)Preconditions.checkNotNull((Object)taskSlot);
        }

        public Task getTask() {
            return this.task;
        }

        public TaskSlot getTaskSlot() {
            return this.taskSlot;
        }
    }
}

