/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.metrics.util;

import com.sun.management.OperatingSystemMXBean;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.lang.management.ClassLoadingMXBean;
import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.ThreadMXBean;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanServer;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.MetricOptions;
import org.apache.flink.metrics.Gauge;
import org.apache.flink.metrics.MetricGroup;
import org.apache.flink.runtime.io.network.NetworkEnvironment;
import org.apache.flink.runtime.metrics.MetricRegistry;
import org.apache.flink.runtime.metrics.groups.JobManagerMetricGroup;
import org.apache.flink.runtime.metrics.groups.TaskManagerMetricGroup;
import org.apache.flink.runtime.taskmanager.TaskManagerLocation;
import org.apache.flink.runtime.util.ProcfsBasedProcessTree;
import org.apache.flink.util.OperatingSystem;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MetricUtils {
    private static final Logger LOG = LoggerFactory.getLogger(MetricUtils.class);
    private static final boolean IS_PROC_FS_AVAILABLE = OperatingSystem.isLinux();

    private MetricUtils() {
    }

    public static JobManagerMetricGroup instantiateJobManagerMetricGroup(MetricRegistry metricRegistry, String hostname) {
        JobManagerMetricGroup jobManagerMetricGroup = new JobManagerMetricGroup(metricRegistry, hostname);
        MetricGroup statusGroup = jobManagerMetricGroup.addGroup("Status");
        MetricUtils.instantiateStatusMetrics(statusGroup);
        return jobManagerMetricGroup;
    }

    public static TaskManagerMetricGroup instantiateTaskManagerMetricGroup(MetricRegistry metricRegistry, TaskManagerLocation taskManagerLocation, NetworkEnvironment network, Configuration configuration) throws IOException {
        TaskManagerMetricGroup taskManagerMetricGroup = new TaskManagerMetricGroup(metricRegistry, taskManagerLocation.getHostname(), taskManagerLocation.getResourceID().toString());
        MetricGroup statusGroup = taskManagerMetricGroup.addGroup("Status");
        MetricUtils.instantiateStatusMetrics(statusGroup);
        MetricGroup networkGroup = statusGroup.addGroup("Network");
        MetricUtils.instantiateNetworkMetrics(networkGroup, network);
        boolean enableProcessTreeMetrics = configuration.getBoolean(MetricOptions.PROCESS_TREE_TM_METRICS_ENABLED);
        if (enableProcessTreeMetrics) {
            MetricUtils.instantiateProcessTreeMetrics(taskManagerMetricGroup, statusGroup, configuration);
        }
        return taskManagerMetricGroup;
    }

    private static void instantiateProcessTreeMetrics(TaskManagerMetricGroup taskManagerMetricGroup, MetricGroup statusGroup, Configuration configuration) throws IOException {
        if (!IS_PROC_FS_AVAILABLE) {
            LOG.info("Only proc filesystem based operating system supports process tree metrics");
            return;
        }
        ScheduledExecutorService scheduledExecutorService = taskManagerMetricGroup.getMetricExecutor();
        boolean smapsEnabled = configuration.getBoolean(MetricOptions.PROCESS_TREE_TM_METRICS_SMAPS_BASED_ENABLED);
        long updateInterval = configuration.getLong(MetricOptions.PROCESS_TREE_TM_METRICS_UPDATE_INTERVAL);
        ProcessTreeUpdater updater = new ProcessTreeUpdater(smapsEnabled);
        scheduledExecutorService.scheduleAtFixedRate(updater, updateInterval, updateInterval, TimeUnit.MILLISECONDS);
        MetricGroup processTreeMetricGroup = statusGroup.addGroup("ProcessTree");
        processTreeMetricGroup.gauge("ProcessCount", () -> updater.processCountOfProcessTree);
        MetricGroup processTreeCPUMetricGroup = processTreeMetricGroup.addGroup("CPU");
        processTreeCPUMetricGroup.gauge("Usage", () -> Float.valueOf(updater.processTreeCPUUsage));
        MetricGroup processTreeMemoryMetricGroup = processTreeMetricGroup.addGroup("Memory");
        processTreeMemoryMetricGroup.gauge("RSS", () -> updater.processTreeMemoryRSS);
        processTreeMemoryMetricGroup.gauge("VIRT", () -> updater.processTreeMemoryVIRT);
    }

    public static void instantiateStatusMetrics(MetricGroup metricGroup) {
        MetricGroup jvm = metricGroup.addGroup("JVM");
        MetricUtils.instantiateClassLoaderMetrics(jvm.addGroup("ClassLoader"));
        MetricUtils.instantiateGarbageCollectorMetrics(jvm.addGroup("GarbageCollector"));
        MetricUtils.instantiateMemoryMetrics(jvm.addGroup("Memory"));
        MetricUtils.instantiateThreadMetrics(jvm.addGroup("Threads"));
        MetricUtils.instantiateCPUMetrics(jvm.addGroup("CPU"));
    }

    private static void instantiateNetworkMetrics(MetricGroup metrics, final NetworkEnvironment network) {
        metrics.gauge("TotalMemorySegments", (Gauge)new Gauge<Long>(){

            public Long getValue() {
                return network.getNetworkBufferPool().getTotalNumberOfMemorySegments();
            }
        });
        metrics.gauge("AvailableMemorySegments", (Gauge)new Gauge<Long>(){

            public Long getValue() {
                return network.getNetworkBufferPool().getNumberOfAvailableMemorySegments();
            }
        });
    }

    private static void instantiateClassLoaderMetrics(MetricGroup metrics) {
        final ClassLoadingMXBean mxBean = ManagementFactory.getClassLoadingMXBean();
        metrics.gauge("ClassesLoaded", (Gauge)new Gauge<Long>(){

            public Long getValue() {
                return mxBean.getTotalLoadedClassCount();
            }
        });
        metrics.gauge("ClassesUnloaded", (Gauge)new Gauge<Long>(){

            public Long getValue() {
                return mxBean.getUnloadedClassCount();
            }
        });
    }

    private static void instantiateGarbageCollectorMetrics(MetricGroup metrics) {
        List<GarbageCollectorMXBean> garbageCollectors = ManagementFactory.getGarbageCollectorMXBeans();
        for (final GarbageCollectorMXBean garbageCollector : garbageCollectors) {
            MetricGroup gcGroup = metrics.addGroup(garbageCollector.getName());
            gcGroup.gauge("Count", (Gauge)new Gauge<Long>(){

                public Long getValue() {
                    return garbageCollector.getCollectionCount();
                }
            });
            gcGroup.gauge("Time", (Gauge)new Gauge<Long>(){

                public Long getValue() {
                    return garbageCollector.getCollectionTime();
                }
            });
        }
    }

    private static void instantiateMemoryMetrics(MetricGroup metrics) {
        final MemoryMXBean mxBean = ManagementFactory.getMemoryMXBean();
        MetricGroup heap = metrics.addGroup("Heap");
        heap.gauge("Used", (Gauge)new Gauge<Long>(){

            public Long getValue() {
                return mxBean.getHeapMemoryUsage().getUsed();
            }
        });
        heap.gauge("Committed", (Gauge)new Gauge<Long>(){

            public Long getValue() {
                return mxBean.getHeapMemoryUsage().getCommitted();
            }
        });
        heap.gauge("Max", (Gauge)new Gauge<Long>(){

            public Long getValue() {
                return mxBean.getHeapMemoryUsage().getMax();
            }
        });
        MetricGroup nonHeap = metrics.addGroup("NonHeap");
        nonHeap.gauge("Used", (Gauge)new Gauge<Long>(){

            public Long getValue() {
                return mxBean.getNonHeapMemoryUsage().getUsed();
            }
        });
        nonHeap.gauge("Committed", (Gauge)new Gauge<Long>(){

            public Long getValue() {
                return mxBean.getNonHeapMemoryUsage().getCommitted();
            }
        });
        nonHeap.gauge("Max", (Gauge)new Gauge<Long>(){

            public Long getValue() {
                return mxBean.getNonHeapMemoryUsage().getMax();
            }
        });
        if (IS_PROC_FS_AVAILABLE) {
            MetricGroup process = metrics.addGroup("Process");
            process.gauge("RSS", ProcessMemCollector::getRssMem);
            process.gauge("Total", ProcessMemCollector::getTotalMem);
        }
        MBeanServer con = ManagementFactory.getPlatformMBeanServer();
        String directBufferPoolName = "java.nio:type=BufferPool,name=direct";
        try {
            ObjectName directObjectName = new ObjectName("java.nio:type=BufferPool,name=direct");
            MetricGroup direct = metrics.addGroup("Direct");
            direct.gauge("Count", new AttributeGauge(con, directObjectName, "Count", -1L));
            direct.gauge("MemoryUsed", new AttributeGauge(con, directObjectName, "MemoryUsed", -1L));
            direct.gauge("TotalCapacity", new AttributeGauge(con, directObjectName, "TotalCapacity", -1L));
        }
        catch (MalformedObjectNameException e) {
            LOG.warn("Could not create object name {}.", (Object)"java.nio:type=BufferPool,name=direct", (Object)e);
        }
        String mappedBufferPoolName = "java.nio:type=BufferPool,name=mapped";
        try {
            ObjectName mappedObjectName = new ObjectName("java.nio:type=BufferPool,name=mapped");
            MetricGroup mapped = metrics.addGroup("Mapped");
            mapped.gauge("Count", new AttributeGauge(con, mappedObjectName, "Count", -1L));
            mapped.gauge("MemoryUsed", new AttributeGauge(con, mappedObjectName, "MemoryUsed", -1L));
            mapped.gauge("TotalCapacity", new AttributeGauge(con, mappedObjectName, "TotalCapacity", -1L));
        }
        catch (MalformedObjectNameException e) {
            LOG.warn("Could not create object name {}.", (Object)"java.nio:type=BufferPool,name=mapped", (Object)e);
        }
    }

    private static void instantiateThreadMetrics(MetricGroup metrics) {
        final ThreadMXBean mxBean = ManagementFactory.getThreadMXBean();
        metrics.gauge("Count", (Gauge)new Gauge<Integer>(){

            public Integer getValue() {
                return mxBean.getThreadCount();
            }
        });
    }

    private static void instantiateCPUMetrics(MetricGroup metrics) {
        try {
            final OperatingSystemMXBean mxBean = (OperatingSystemMXBean)ManagementFactory.getOperatingSystemMXBean();
            metrics.gauge("Load", (Gauge)new Gauge<Double>(){

                public Double getValue() {
                    return mxBean.getProcessCpuLoad();
                }
            });
            metrics.gauge("Time", (Gauge)new Gauge<Long>(){

                public Long getValue() {
                    return mxBean.getProcessCpuTime();
                }
            });
            if (IS_PROC_FS_AVAILABLE) {
                metrics.gauge("Usage", (Gauge)new Gauge<Double>(){

                    public Double getValue() {
                        return ProcessCpuCollector.getUsage();
                    }
                });
            }
        }
        catch (Exception e) {
            LOG.warn("Cannot access com.sun.management.OperatingSystemMXBean.getProcessCpuLoad() - CPU load metrics will not be available.", (Throwable)e);
        }
    }

    private static class ProcessTreeUpdater
    implements Runnable {
        private ProcfsBasedProcessTree processTree;
        private int processCountOfProcessTree = 0;
        private float processTreeCPUUsage = 0.0f;
        private long processTreeMemoryRSS = 0L;
        private long processTreeMemoryVIRT = 0L;

        public ProcessTreeUpdater(boolean smapsEnabled) throws IOException {
            this.processTree = new ProcfsBasedProcessTree(smapsEnabled);
        }

        @Override
        public void run() {
            try {
                this.processTree.updateProcessTree();
                this.processCountOfProcessTree = this.processTree.getCurrentProcessIDs().size();
                float cpuUsagePercent = this.processTree.getCpuUsagePercent();
                this.processTreeCPUUsage = cpuUsagePercent > 0.0f ? cpuUsagePercent / 100.0f : cpuUsagePercent;
                this.processTreeMemoryRSS = this.processTree.getRssMemorySize();
                this.processTreeMemoryVIRT = this.processTree.getVirtualMemorySize();
            }
            catch (Throwable t) {
                LOG.warn("Updating metric process tree failed", t);
            }
        }
    }

    public static class ProcessMemCollector {
        private static final Logger LOG = LoggerFactory.getLogger(ProcessMemCollector.class);
        private static final double KILO = 1024.0;
        private static final String MEMORY_TOTAL_USE_TOKEN = "VmSize";
        private static final String MEMORY_RSS_USE_TOKEN = "VmRSS";
        private static final String MEM_STAT_FILE = "/proc/self/status";

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public static long getTotalMem() {
            BufferedReader br = null;
            try {
                String line;
                br = new BufferedReader(new FileReader(MEM_STAT_FILE));
                while (null != (line = br.readLine())) {
                    if (!line.startsWith(MEMORY_TOTAL_USE_TOKEN)) continue;
                    long l = ProcessMemCollector.getNumber(line);
                    return l;
                }
            }
            catch (IOException ex) {
                LOG.warn("collect mem use exception " + ex.getMessage());
            }
            finally {
                if (br != null) {
                    try {
                        br.close();
                    }
                    catch (IOException iOException) {}
                }
            }
            return 0L;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public static long getRssMem() {
            BufferedReader br = null;
            try {
                String line;
                br = new BufferedReader(new FileReader(MEM_STAT_FILE));
                while (null != (line = br.readLine())) {
                    if (!line.startsWith(MEMORY_RSS_USE_TOKEN)) continue;
                    long l = ProcessMemCollector.getNumber(line);
                    return l;
                }
            }
            catch (IOException ex) {
                LOG.warn("collect mem use exception " + ex.getMessage());
            }
            finally {
                if (br != null) {
                    try {
                        br.close();
                    }
                    catch (IOException iOException) {}
                }
            }
            return 0L;
        }

        private static long getNumber(String line) {
            int beginIndex = line.indexOf(":") + 1;
            int endIndex = line.indexOf("kB") - 1;
            String memSize = line.substring(beginIndex, endIndex).trim();
            return 1024L * Long.parseLong(memSize);
        }
    }

    public static class ProcessCpuCollector {
        private static final Logger LOG = LoggerFactory.getLogger(ProcessCpuCollector.class);
        private static final String CPU_STAT_FILE = "/proc/stat";
        private static final String PROCESS_STAT_FILE = "/proc/self/stat";
        private static final int AVAILABLE_PROCESSOR_COUNT = Runtime.getRuntime().availableProcessors();
        private static double currentSystemCpuTotal = 0.0;
        private static double currentProcCpuClock = 0.0;

        public static double getUsage() {
            try {
                double procCpuClock = ProcessCpuCollector.getProcessCpuClock();
                double totalCpuStat = ProcessCpuCollector.getTotalCpuClock();
                if (totalCpuStat == 0.0) {
                    return 0.0;
                }
                return procCpuClock / totalCpuStat * (double)AVAILABLE_PROCESSOR_COUNT;
            }
            catch (IOException ex) {
                LOG.warn("collect cpu load exception " + ex.getMessage());
                return 0.0;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private static String getFirstLineFromFile(String fileName) throws IOException {
            try (BufferedReader br = null;){
                String line;
                br = new BufferedReader(new FileReader(fileName));
                String string = line = br.readLine();
                return string;
            }
        }

        private static double getProcessCpuClock() throws IOException {
            double lastProcCpuClock = currentProcCpuClock;
            String content = ProcessCpuCollector.getFirstLineFromFile(PROCESS_STAT_FILE);
            if (content == null) {
                throw new IOException("read /proc/self/stat null !");
            }
            String[] processStats = content.split(" ", -1);
            if (processStats.length < 17) {
                LOG.error("parse cpu stat file failed! the first line is:" + content);
                throw new IOException("parse process stat file failed!");
            }
            int rBracketPos = 0;
            for (int i = processStats.length - 1; i > 0; --i) {
                if (!processStats[i].contains(")")) continue;
                rBracketPos = i;
                break;
            }
            if (rBracketPos == 0) {
                throw new IOException("get right bracket pos error");
            }
            double cpuTotal = 0.0;
            for (int i = rBracketPos + 12; i < rBracketPos + 16; ++i) {
                cpuTotal += Double.parseDouble(processStats[i]);
            }
            currentProcCpuClock = cpuTotal;
            return currentProcCpuClock - lastProcCpuClock;
        }

        private static double getTotalCpuClock() throws IOException {
            double lastSystemCpuTotal = currentSystemCpuTotal;
            String line = ProcessCpuCollector.getFirstLineFromFile(CPU_STAT_FILE);
            if (line == null) {
                throw new IOException("read /proc/stat null !");
            }
            String[] cpuStats = line.split(" ", -1);
            if (cpuStats.length < 11) {
                LOG.error("parse cpu stat file failed! the first line is:" + line);
                throw new IOException("parse cpu stat file failed!");
            }
            double statCpuTotal = 0.0;
            for (int i = 2; i < cpuStats.length; ++i) {
                statCpuTotal += Double.parseDouble(cpuStats[i]);
            }
            currentSystemCpuTotal = statCpuTotal;
            return currentSystemCpuTotal - lastSystemCpuTotal;
        }
    }

    private static final class AttributeGauge<T>
    implements Gauge<T> {
        private final MBeanServer server;
        private final ObjectName objectName;
        private final String attributeName;
        private final T errorValue;

        private AttributeGauge(MBeanServer server, ObjectName objectName, String attributeName, T errorValue) {
            this.server = (MBeanServer)Preconditions.checkNotNull((Object)server);
            this.objectName = (ObjectName)Preconditions.checkNotNull((Object)objectName);
            this.attributeName = (String)Preconditions.checkNotNull((Object)attributeName);
            this.errorValue = errorValue;
        }

        public T getValue() {
            try {
                return (T)this.server.getAttribute(this.objectName, this.attributeName);
            }
            catch (AttributeNotFoundException | InstanceNotFoundException | MBeanException | ReflectionException e) {
                LOG.warn("Could not read attribute {}.", (Object)this.attributeName, (Object)e);
                return this.errorValue;
            }
        }
    }
}

