/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.streaming.runtime.streamrecord;

import org.apache.flink.core.memory.DataInputView;
import org.apache.flink.core.memory.DataOutputView;
import org.apache.flink.runtime.plugable.DeserializationDelegate;

import java.io.IOException;

/**
 * A {@link DeserializationDelegate} that reuses an existing record-value instance upon deserialization.
 */
public class ReusingRecordValueDeserializationDelegate<IN> implements DeserializationDelegate<StreamElement> {

	private StreamElement instance;

	private IN lastReusableRecordValueObj;

	private final StreamElementSerializer<IN> serializer;

	public ReusingRecordValueDeserializationDelegate(StreamElementSerializer<IN> serializer) {
		this.serializer = serializer;
	}

	@Override
	public void setInstance(StreamElement instance) {
		this.instance = instance;
		updateLastReusableRecordValueObj();
	}

	@Override
	public StreamElement getInstance() {
		return instance;
	}

	@Override
	public void write(DataOutputView out) throws IOException {
		throw new IllegalStateException("Serialization method called on DeserializationDelegate.");
	}

	@Override
	public void read(DataInputView in) throws IOException {
		this.instance = (lastReusableRecordValueObj == null) ?
					serializer.deserialize(in) : serializer.deserialize(new StreamRecord<>(lastReusableRecordValueObj), in);

		updateLastReusableRecordValueObj();
	}

	private void updateLastReusableRecordValueObj() {
		if (instance != null && instance.isRecord()) {
			StreamRecord<IN> record = instance.asRecord();
			if (record.getValue() != null) {
				lastReusableRecordValueObj = record.getValue();
			}
		}
	}
}
