/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.streaming.api.driver;

import org.apache.flink.api.common.DriverProgram;
import org.apache.flink.api.common.operators.ResourceSpec;
import org.apache.flink.api.common.restartstrategy.RestartStrategies;
import org.apache.flink.client.program.PackagedProgram;
import org.apache.flink.configuration.Configuration;
import org.apache.flink.configuration.DriverConfigConstants;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.flink.streaming.api.functions.sink.SinkFunction;
import org.apache.flink.streaming.api.functions.source.SourceFunction;

import org.apache.commons.io.FilenameUtils;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.List;
import java.util.Map;

/**
 * DriverEntry is the main entry point of driver mode. It manages package program user provided and provides a driver
 * environment to run user application by preparing a flink source function in which will call the main method of
 * user application.
 */
public class DriverEntry implements DriverProgram {

	public static final String BLOB_TYPE_SEPARATOR = "#";

	public static final String BLOB_TYPE_JARFILE = "jarFile";

	public static final String BLOB_TYPE_EXTERNAL_FILE = "externalFile";

	public static final String BLOB_TYPE_CLASSPATH = "classPath";

	public static final String BLOB_TYPE_LIB_JARS = "libJars";

	private static PackagedProgram packagedProgram;

	private static Configuration configuration;

	private static String driveName;

	@Override
	public void setParameter(Object parameter) {
		if (parameter == null || parameter instanceof PackagedProgram) {
			packagedProgram = (PackagedProgram) parameter;
		}
	}

	@Override
	public void setConfiguration(Configuration configuration) {
		this.configuration = configuration;
	}

	public static void main(String[] args) throws Exception {
		driveName = "FlinkDriverJob" + "_" + System.currentTimeMillis();
		StreamExecutionEnvironment env = StreamExecutionEnvironment.getExecutionEnvironment();
		manageJobBlobs(env, packagedProgram.getAllLibraries(), BLOB_TYPE_JARFILE);
		manageJobBlobs(env, packagedProgram.getFiles(), BLOB_TYPE_EXTERNAL_FILE);
		manageJobBlobs(env, packagedProgram.getClasspaths(), BLOB_TYPE_CLASSPATH);
		manageJobBlobs(env, packagedProgram.getLibjars(), BLOB_TYPE_LIB_JARS);

		if (packagedProgram.getDriverArgs() != null) {
			ResourceSpec resourceSpec = buildDriverResourceSpec(packagedProgram.getDriverArgs());
			env.setDefaultResources(resourceSpec);
		}
		env.setRestartStrategy(RestartStrategies.noRestart());
		env.getCustomConfiguration().setInteger(DriverConfigConstants.FLINK_DRIVER_MANAGE_PRIORIRY,
			DriverConfigConstants.FLINK_DRIVER_MANAGE_PRIORITY_DRIVER);
		env.addSource(createSourceFunction()).addSink(new SinkFunction<Byte>() {
			@Override
			public void invoke(Byte value, Context context) throws Exception {

			}
		});
		env.submit(driveName);
	}

	private static SourceFunction<Byte> createSourceFunction() {
		DriverSourceFunction driverSourceFunction = new DriverSourceFunction(
			driveName,
			packagedProgram.getDriverClass(),
			packagedProgram.getArguments(),
			packagedProgram.getAllLibraries(),
			packagedProgram.getClasspaths(),
			DriverSourceFunction.transURItoURL(packagedProgram.getLibjars()),
			DriverSourceFunction.transURItoURL(packagedProgram.getFiles()),
			configuration);
		return driverSourceFunction;
	}

	private static void manageJobBlobs(StreamExecutionEnvironment env, List<? extends Object> jobBlobs, String suffix) {
		for (Object blod : jobBlobs) {
			URL url = null;
			if (blod instanceof URI) {
				try {
					url = ((URI) blod).toURL();
				} catch (MalformedURLException e) {
					e.printStackTrace();
				}
			} else if (blod instanceof URL) {
				url = (URL) blod;
			}
			if (url != null) {
				String fileName = FilenameUtils.getName(url.getPath());
				env.registerCachedFile(url.toString(), suffix + BLOB_TYPE_SEPARATOR + fileName);
			}
		}
	}

	private static ResourceSpec buildDriverResourceSpec(Map<String, Object> args) {
		ResourceSpec.Builder builder = ResourceSpec.newBuilder();
		if (args.containsKey(DriverConfigConstants.FLINK_DRIVER_CPU_CORE_KEY)) {
			builder.setCpuCores((double) (args.get(DriverConfigConstants.FLINK_DRIVER_CPU_CORE_KEY)));
		}

		if (args.containsKey(DriverConfigConstants.FLINK_DRIVER_HEAP_MEMORY_KEY)) {
			builder.setHeapMemoryInMB((int) (args.get(DriverConfigConstants.FLINK_DRIVER_HEAP_MEMORY_KEY)));
		}

		if (args.containsKey(DriverConfigConstants.FLINK_DRIVER_DIRECT_MEMORY_KEY)) {
			builder.setDirectMemoryInMB((int) (args.get(DriverConfigConstants.FLINK_DRIVER_DIRECT_MEMORY_KEY)));
		}

		if (args.containsKey(DriverConfigConstants.FLINK_DRIVER_NATIVE_MEMORY_KEY)) {
			builder.setNativeMemoryInMB((int) (args.get(DriverConfigConstants.FLINK_DRIVER_NATIVE_MEMORY_KEY)));
		}
		return builder.build();
	}

}
