/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.streaming.runtime.tasks;

import org.apache.flink.annotation.Internal;
import org.apache.flink.api.java.tuple.Tuple3;
import org.apache.flink.runtime.checkpoint.CheckpointMetaData;
import org.apache.flink.runtime.checkpoint.CheckpointOptions;
import org.apache.flink.runtime.checkpoint.decline.CheckpointDeclineTaskNotReadyException;
import org.apache.flink.runtime.execution.Environment;
import org.apache.flink.streaming.api.TimeCharacteristic;
import org.apache.flink.streaming.api.functions.source.SourceFunction;
import org.apache.flink.streaming.api.functions.source.SourceFunctionV2;
import org.apache.flink.streaming.api.functions.source.SourceRecord;
import org.apache.flink.streaming.api.operators.OneInputStreamOperator;
import org.apache.flink.streaming.api.operators.Output;
import org.apache.flink.streaming.api.operators.StreamOperator;
import org.apache.flink.streaming.api.operators.StreamSourceContexts;
import org.apache.flink.streaming.api.operators.StreamSourceV2;
import org.apache.flink.streaming.api.watermark.Watermark;
import org.apache.flink.streaming.runtime.streamrecord.StreamRecord;
import org.apache.flink.util.Preconditions;
import org.apache.flink.util.checkpointlock.CheckpointLockDelegate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayDeque;

/**
 * {@link StreamTask} for executing a {@link StreamSourceV2}.
 *
 * @param <OUT> Type of the output elements of this source.
 * @param <SRC> Type of the source function for the stream source operator
 * @param <OP> Type of the stream source operator
 */
@Internal
public class SourceStreamTaskV2<OUT, SRC extends SourceFunctionV2<OUT>, OP extends StreamSourceV2<OUT, SRC>>
	extends StreamTask<OUT, OP> {

	private static final Logger LOG = LoggerFactory.getLogger(SourceStreamTaskV2.class);

	private final ArrayDeque<Tuple3<CheckpointMetaData, CheckpointOptions, Boolean>> pendingCheckpoints;

	private volatile boolean checkpointTriggered = false;

	private volatile boolean running = true;

	public SourceStreamTaskV2(Environment env) {
		super(env);

		pendingCheckpoints = new ArrayDeque<>();
	}

	@Override
	protected void init() {
	}

	@Override
	protected void cleanup() {
		synchronized (pendingCheckpoints) {
			pendingCheckpoints.clear();
			checkpointTriggered = false;
		}
	}

	@Override
	protected void run() throws Exception {

		OP headOperator = getHeadOperator();

		final TimeCharacteristic timeCharacteristic = headOperator.getOperatorConfig().getTimeCharacteristic();

		Output<StreamRecord<OUT>> collector = headOperator.getOutput();

		final long watermarkInterval = headOperator.getRuntimeContext().getExecutionConfig().getAutoWatermarkInterval();

		SourceFunction.SourceContext<OUT> ctx = StreamSourceContexts.getSourceContext(
			timeCharacteristic,
			getProcessingTimeService(),
			getCheckpointLock(),
			getStreamStatusMaintainer(),
			collector,
			watermarkInterval,
			-1);

		try {
			boolean isIdle = false;
			while (running) {
				SourceRecord<OUT> sourceRecord = headOperator.next();
				if (sourceRecord != null) {
					OUT out = sourceRecord.getRecord();
					if (out != null) {
						if (sourceRecord.getTimestamp() > 0) {
							ctx.collectWithTimestamp(out, sourceRecord.getTimestamp());
						} else {
							ctx.collect(out);
						}
					}
					if (sourceRecord.getWatermark() != null) {
						ctx.emitWatermark(sourceRecord.getWatermark());
					}
				} else {
					isIdle = true;
				}
				if (headOperator.isFinished()) {
					break;
				}

				if (checkpointTriggered) {
					isIdle = false;

					final Tuple3<CheckpointMetaData, CheckpointOptions, Boolean> checkpointInfos;
					synchronized (pendingCheckpoints) {
						checkpointInfos = pendingCheckpoints.poll();
						if (pendingCheckpoints.isEmpty()) {
							checkpointTriggered = false;
						}
					}
					if (checkpointInfos != null) {
						if (!super.triggerCheckpoint(checkpointInfos.f0, checkpointInfos.f1, checkpointInfos.f2)) {
							getEnvironment().declineCheckpoint(
								checkpointInfos.f0.getCheckpointId(),
								new CheckpointDeclineTaskNotReadyException(getName()));
						}
					}
				}

				if (isIdle) {
					Thread.sleep(10);
					isIdle = false;
				}
			}

			// if we get here, then the user function either exited after being done (finite source)
			// or the function was canceled or stopped. For the finite source case, we should emit
			// a final watermark that indicates that we reached the end of event-time
			if (running) {
				ctx.emitWatermark(Watermark.MAX_WATERMARK);

				new CheckpointLockDelegate(getCheckpointLock()).lockAndRun(() -> {
					for (StreamOperator operator : operatorChain.getAllOperatorsTopologySorted()) {
						if (operator instanceof OneInputStreamOperator) {
							((OneInputStreamOperator) operator).endInput();
						}
					}
				});
			} else {
				headOperator.cancel();
				// the context may not be initialized if the source was never running.
			}
		} finally {
			// make sure that the context is closed in any case
			if (ctx != null) {
				ctx.close();
				ctx = null;
			}
		}
	}

	@Override
	protected void cancelTask() throws Exception {
		running = false;
	}

	@Override
	public boolean triggerCheckpoint(CheckpointMetaData checkpointMetaData, CheckpointOptions checkpointOptions, boolean advanceToEndOfEventTime) throws Exception {
		if (!running) {
			return false;
		}
		synchronized (pendingCheckpoints) {
			if (pendingCheckpoints.size() >= maxConcurrentCheckpoints) {
				pendingCheckpoints.poll();
			}
			pendingCheckpoints.add(Tuple3.of(checkpointMetaData, checkpointOptions, advanceToEndOfEventTime));
			checkpointTriggered = true;
		}
		return true;
	}

	@SuppressWarnings("unchecked")
	protected OP getHeadOperator() {
		Preconditions.checkState(operatorChain.getHeadOperators().length == 1,
			"There should only one head operator, not " + operatorChain.getHeadOperators().length);
		return (OP) operatorChain.getHeadOperators()[0];
	}
}
