/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.flink.table.plan.rules.physical.stream

import org.apache.flink.table.api.TableException
import org.apache.flink.table.plan.`trait`.FlinkRelDistribution
import org.apache.flink.table.plan.nodes.FlinkConventions
import org.apache.flink.table.plan.nodes.logical.{FlinkLogicalRank, FlinkLogicalSort}
import org.apache.flink.table.plan.nodes.physical.stream.StreamExecRank
import org.apache.flink.table.plan.schema.BaseRowSchema
import org.apache.flink.table.plan.util.ConstantRankRange

import org.apache.calcite.plan.volcano.RelSubset
import org.apache.calcite.plan.{RelOptRule, RelOptRuleCall}
import org.apache.calcite.rel.RelNode
import org.apache.calcite.rel.convert.ConverterRule
import org.apache.calcite.rex.RexLiteral
import org.apache.calcite.sql.fun.SqlStdOperatorTable

object StreamExecRankRules {
  val SORT_INSTANCE: RelOptRule = new StreamExecRankFromSortRule
  val RANK_INSTANCE: RelOptRule = new StreamExecRankFromRankRule

  class StreamExecRankFromSortRule
    extends ConverterRule(
      classOf[FlinkLogicalSort],
      FlinkConventions.LOGICAL,
      FlinkConventions.STREAM_PHYSICAL,
      "StreamExecRankFromSortRule")
    with BaseStreamExecRankRule {

    override def matches(call: RelOptRuleCall): Boolean = {
      val sort: FlinkLogicalSort = call.rel(0)
      val sortCollation = sort.collation

      val canConvertToRank = if (sortCollation.getFieldCollations.isEmpty) {
        true
      } else {
        sort.fetch != null
      }
      canConvertToRank && !canSimplifyToFirstLastRow(sort)
    }

    override def convert(rel: RelNode): RelNode = {
      val sort = rel.asInstanceOf[FlinkLogicalSort]

      val requiredDistribution = FlinkRelDistribution.SINGLETON

      val requiredTraitSet = sort.getInput.getTraitSet
        .replace(FlinkConventions.STREAM_PHYSICAL)
        .replace(requiredDistribution)

      val convInput: RelNode = RelOptRule.convert(sort.getInput, requiredTraitSet)

      val rankStart = if (sort.offset != null) {
        RexLiteral.intValue(sort.offset) + 1
      } else {
        1
      }

      val rankEnd = if (sort.fetch != null) {
        rankStart + RexLiteral.intValue(sort.fetch) - 1
      } else {
        // we have checked in matches method that fetch is not null
        throw new TableException("This should never happen, please file an issue.")
      }

      val providedTraitSet = rel.getTraitSet.replace(FlinkConventions.STREAM_PHYSICAL)

      new StreamExecRank(
        rel.getCluster,
        providedTraitSet,
        convInput,
        new BaseRowSchema(rel.getRowType),
        new BaseRowSchema(rel.getRowType),
        SqlStdOperatorTable.ROW_NUMBER,
        Array(),
        sort.collation,
        ConstantRankRange(rankStart, rankEnd),
        outputRankFunColumn = false)
    }
  }

  class StreamExecRankFromRankRule
    extends ConverterRule(
      classOf[FlinkLogicalRank],
      FlinkConventions.LOGICAL,
      FlinkConventions.STREAM_PHYSICAL,
      "StreamExecRankFromRankRule")
    with BaseStreamExecRankRule {


    override def matches(call: RelOptRuleCall): Boolean = {
      val rank: FlinkLogicalRank = call.rel(0)
      !canSimplifyToFirstLastRow(rank)
    }

    override def convert(rel: RelNode): RelNode = {
      val rank = rel.asInstanceOf[FlinkLogicalRank]

      val requiredDistribution = if (!rank.partitionKey.isEmpty) {
        FlinkRelDistribution.hash(rank.partitionKey.asList())
      } else {
        FlinkRelDistribution.SINGLETON
      }
      val requiredTraitSet = rank.getInput.getTraitSet
        .replace(FlinkConventions.STREAM_PHYSICAL)
        .replace(requiredDistribution)
      val providedTraitSet = rank.getTraitSet.replace(FlinkConventions.STREAM_PHYSICAL)
      val convInput: RelNode = RelOptRule.convert(rank.getInput, requiredTraitSet)
      val inputRowType = convInput.asInstanceOf[RelSubset].getOriginal.getRowType

      new StreamExecRank(
        rank.getCluster,
        providedTraitSet,
        convInput,
        new BaseRowSchema(inputRowType),
        new BaseRowSchema(rank.getRowType),
        rank.rankFunction,
        rank.partitionKey.toArray,
        rank.sortCollation,
        rank.rankRange,
        rank.outputRankFunColumn)
    }
  }

}
