/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.runtime.util;

import java.util.Arrays;
import org.apache.flink.table.dataformat.util.BinaryRowUtil;
import org.apache.flink.util.Preconditions;

public class BloomFilter {
    private static final long MIN_BLOOM_FILTER_ENTRIES = 500000L;
    public static final double DEFAULT_FPP = (double)0.03f;
    private final int numBits;
    private final int numHashFunctions;
    private final BitSet bitSet;

    public BloomFilter(long maxNumEntries) {
        this(maxNumEntries, 0.03f);
    }

    public BloomFilter(long maxNumEntries, double fpp) {
        Preconditions.checkArgument(maxNumEntries > 0L, "expectedEntries should be > 0");
        int nb = BloomFilter.optimalNumOfBits(maxNumEntries, fpp);
        this.numBits = nb + (64 - nb % 64);
        this.numHashFunctions = BloomFilter.optimalNumOfHashFunctions(maxNumEntries, this.numBits);
        this.bitSet = new BitSet(this.numBits);
    }

    private BloomFilter(long[] bits, int numFuncs) {
        this.numBits = bits.length * 64;
        this.numHashFunctions = numFuncs;
        this.bitSet = new BitSet(bits);
    }

    public static long getLongHash(long key) {
        key = (key ^ 0xFFFFFFFFFFFFFFFFL) + (key << 21);
        key ^= key >> 24;
        key = key + (key << 3) + (key << 8);
        key ^= key >> 14;
        key = key + (key << 2) + (key << 4);
        key ^= key >> 28;
        key += key << 31;
        return key;
    }

    public static long getLongHash(double key) {
        return BloomFilter.getLongHash(Double.doubleToLongBits(key));
    }

    public void addHash(long hash64) {
        int hash1 = (int)hash64;
        int hash2 = (int)(hash64 >>> 32);
        for (int i = 1; i <= this.numHashFunctions; ++i) {
            int combinedHash = hash1 + (i + 1) * hash2;
            if (combinedHash < 0) {
                combinedHash ^= 0xFFFFFFFF;
            }
            int pos = combinedHash % this.numBits;
            this.bitSet.set(pos);
        }
    }

    public boolean testHash(long hash64) {
        int hash1 = (int)hash64;
        int hash2 = (int)(hash64 >>> 32);
        for (int i = 1; i <= this.numHashFunctions; ++i) {
            int pos;
            int combinedHash = hash1 + (i + 1) * hash2;
            if (combinedHash < 0) {
                combinedHash ^= 0xFFFFFFFF;
            }
            if (this.bitSet.get(pos = combinedHash % this.numBits)) continue;
            return false;
        }
        return true;
    }

    public long[] getBitSet() {
        return this.bitSet.getData();
    }

    public void merge(BloomFilter that) {
        if (this == that || this.numBits != that.numBits || this.numHashFunctions != that.numHashFunctions) {
            throw new IllegalArgumentException("BloomKFilters are not compatible for merging. this - " + this.toString() + " that - " + that.toString());
        }
        this.bitSet.putAll(that.bitSet);
    }

    public void reset() {
        this.bitSet.clear();
    }

    public String toString() {
        return "numBits: " + this.numBits + " numHashFunctions: " + this.numHashFunctions;
    }

    public static byte[] toBytes(BloomFilter filter) {
        long[] bitSet = filter.getBitSet();
        int longLen = bitSet.length;
        byte[] bytes = new byte[5 + longLen * 8];
        BinaryRowUtil.UNSAFE.putByte(bytes, BinaryRowUtil.BYTE_ARRAY_BASE_OFFSET, (byte)filter.numHashFunctions);
        BinaryRowUtil.UNSAFE.putInt(bytes, BinaryRowUtil.BYTE_ARRAY_BASE_OFFSET + 1, longLen);
        BinaryRowUtil.UNSAFE.copyMemory(bitSet, BinaryRowUtil.LONG_ARRAY_OFFSET, bytes, BinaryRowUtil.BYTE_ARRAY_BASE_OFFSET + 5, longLen * 8);
        return bytes;
    }

    public static BloomFilter fromBytes(byte[] bytes) {
        byte numHashFunc = BinaryRowUtil.UNSAFE.getByte(bytes, BinaryRowUtil.BYTE_ARRAY_BASE_OFFSET);
        int longLen = BinaryRowUtil.UNSAFE.getInt(bytes, BinaryRowUtil.BYTE_ARRAY_BASE_OFFSET + 1);
        long[] data = new long[longLen];
        BinaryRowUtil.UNSAFE.copyMemory(bytes, BinaryRowUtil.BYTE_ARRAY_BASE_OFFSET + 5, data, BinaryRowUtil.LONG_ARRAY_OFFSET, longLen * 8);
        return new BloomFilter(data, numHashFunc);
    }

    public static long suitableMaxNumEntries(long maxNumEntries) {
        return Math.max(maxNumEntries, 500000L);
    }

    private static int optimalNumOfHashFunctions(long n, long m) {
        return Math.max(1, (int)Math.round((double)m / (double)n * Math.log(2.0)));
    }

    private static int optimalNumOfBits(long maxNumEntries, double fpp) {
        return (int)((double)(-maxNumEntries) * Math.log(fpp) / (Math.log(2.0) * Math.log(2.0)));
    }

    public static double findSuitableFpp(long entries, double maxNumOfBits) {
        for (double f = (double)0.03f; f < 1.0; f += (double)0.01f) {
            long bits = BloomFilter.optimalNumOfBits(entries, f);
            if (!((double)bits < maxNumOfBits)) continue;
            return f;
        }
        return 1.0;
    }

    public static void mergeBloomFilterBytes(byte[] bf1Bytes, byte[] bf2Bytes) {
        BloomFilter.mergeBloomFilterBytes(bf1Bytes, 0, bf1Bytes.length, bf2Bytes, 0, bf2Bytes.length);
    }

    public static void mergeBloomFilterBytes(byte[] bf1Bytes, int bf1Start, int bf1Length, byte[] bf2Bytes, int bf2Start, int bf2Length) {
        if (bf1Length != bf2Length) {
            throw new IllegalArgumentException("bf1Length " + bf1Length + " does not match bf2Length " + bf2Length);
        }
        int longLen1 = BinaryRowUtil.UNSAFE.getInt(bf1Bytes, BinaryRowUtil.BYTE_ARRAY_BASE_OFFSET + bf1Start + 1);
        if (BinaryRowUtil.UNSAFE.getByte(bf1Bytes, BinaryRowUtil.BYTE_ARRAY_BASE_OFFSET + bf1Start) != BinaryRowUtil.UNSAFE.getByte(bf2Bytes, BinaryRowUtil.BYTE_ARRAY_BASE_OFFSET + bf2Start) || longLen1 != BinaryRowUtil.UNSAFE.getInt(bf2Bytes, BinaryRowUtil.BYTE_ARRAY_BASE_OFFSET + bf2Start + 1)) {
            throw new IllegalArgumentException("bf1 NumHashFunctions/NumBits does not match bf2");
        }
        for (int idx = 5 + BinaryRowUtil.BYTE_ARRAY_BASE_OFFSET; idx < bf1Length + BinaryRowUtil.BYTE_ARRAY_BASE_OFFSET; idx += 8) {
            long l1 = BinaryRowUtil.UNSAFE.getLong(bf1Bytes, bf1Start + idx);
            long l2 = BinaryRowUtil.UNSAFE.getLong(bf2Bytes, bf2Start + idx);
            BinaryRowUtil.UNSAFE.putLong(bf1Bytes, bf1Start + idx, l1 | l2);
        }
    }

    public static class BitSet {
        private final long[] data;

        BitSet(long bits) {
            this(new long[(int)Math.ceil((double)bits / 64.0)]);
        }

        BitSet(long[] data) {
            assert (data.length > 0) : "data length is zero!";
            this.data = data;
        }

        public void set(int index) {
            int n = index >>> 6;
            this.data[n] = this.data[n] | 1L << index;
        }

        public boolean get(int index) {
            return (this.data[index >>> 6] & 1L << index) != 0L;
        }

        public long[] getData() {
            return this.data;
        }

        public void putAll(BitSet array2) {
            assert (this.data.length == array2.data.length) : "BitArrays must be of equal length (" + this.data.length + "!= " + array2.data.length + ")";
            for (int i = 0; i < this.data.length; ++i) {
                int n = i;
                this.data[n] = this.data[n] | array2.data[i];
            }
        }

        public void clear() {
            Arrays.fill(this.data, 0L);
        }
    }
}

