/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.api;

import org.apache.flink.table.types.IntType;
import org.apache.flink.table.types.InternalType;
import org.apache.flink.table.types.StringType;

/**
 * Virtual column appended by flink to support delete and only when delete dml.
 * If a user writes a table, flink will append row_id and file_name to record the position of every
 * row. When we want to delete a row, we only need to provide the row_id and file_name.
 *
 */
public enum VirtualColumn {

	ROWID("ROW__ID", IntType.INSTANCE, false),
	FILENAME("INPUT__FILE__NAME", StringType.INSTANCE, false);

	private final String name;

	public boolean isNullable() {
		return isNullable;
	}

	private final boolean isNullable;

	public String getName() {
		return name;
	}

	public InternalType getInternalType() {
		return internalType;
	}

	private final InternalType internalType;

	VirtualColumn(String name, InternalType internalType, boolean isNullable) {
		this.name = name;
		this.internalType = internalType;
		this.isNullable = isNullable;
	}

	public static String[] getNames() {
		return new String[]{ROWID.name, FILENAME.name};
	}

	public static InternalType[] getTypes() {
		return new InternalType[]{ROWID.internalType, FILENAME.internalType};
	}

	public static boolean[] getNullables() {
		return new boolean[]{ROWID.isNullable, FILENAME.isNullable};
	}

	public static VirtualColumn[] getVirtualColumns() {
		return new VirtualColumn[]{ROWID, FILENAME};
	}
}
