/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.calcite;

import org.apache.flink.table.catalog.CatalogManager;
import org.apache.flink.table.catalog.ObjectPath;
import org.apache.flink.table.catalog.ReadableCatalog;
import org.apache.flink.table.validate.FunctionCatalogUtils;

import org.apache.calcite.sql.SqlFunction;
import org.apache.calcite.sql.SqlFunctionCategory;
import org.apache.calcite.sql.SqlIdentifier;
import org.apache.calcite.sql.SqlOperator;
import org.apache.calcite.sql.SqlOperatorTable;
import org.apache.calcite.sql.SqlSyntax;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * Implementation of the {@link SqlOperatorTable} interface by
 * lookup {@link SqlOperator operators} on demand.
 */
public class LazySqlOperatorTable implements SqlOperatorTable {

	private final CatalogManager catalogManager;
	private final FlinkTypeFactory typeFactory;

	public LazySqlOperatorTable(CatalogManager catalogManager, FlinkTypeFactory typeFactory) {
		this.catalogManager = catalogManager;
		this.typeFactory = typeFactory;
	}

	@Override
	public void lookupOperatorOverloads(
		SqlIdentifier opName,
		SqlFunctionCategory category,
		SqlSyntax syntax,
		List<SqlOperator> operatorList) {

		if (syntax != SqlSyntax.FUNCTION) {
			return;
		}

		if (!opName.isSimple()) {
			return;
		}

		if (category == null
			|| !category.isUserDefinedNotSpecificFunction()) {
			return;
		}

		getFunctionsFrom(opName.names)
			.stream()
			.forEachOrdered(operatorList::add);
	}

	private Collection<SqlFunction> getFunctionsFrom(List<String> names) {
		final List<SqlFunction> sqlFunctions = new ArrayList<>();
		if (names.size() > 1) {
			// functions with qualified names
			// TODO: support namespace
		} else {
			final ReadableCatalog catalog = catalogManager.getDefaultCatalog();
			final String funcName = names.get(0).toLowerCase();
			final ObjectPath functionPath = new ObjectPath(catalogManager.getDefaultDatabaseName(), funcName);
			if (catalog.functionExists(functionPath)) {
				sqlFunctions.add(
					FunctionCatalogUtils.toSqlFunction(
						catalog, functionPath.getObjectName(), catalog.getFunction(functionPath), typeFactory)
				);
			}
		}
		return sqlFunctions;
	}

	@Override
	public List<SqlOperator> getOperatorList() {
		// TODO: getOperatorList is designed for automated testing
		// We should not use it. See TableEnvironment::listUserDefinedFunctions (line 1202)
		final List<SqlOperator> sqlOperators = new ArrayList<>();
		final ReadableCatalog catalog = catalogManager.getDefaultCatalog();
		final String defaultDatabase = catalogManager.getDefaultDatabaseName();
		catalog
			.listFunctions(defaultDatabase)
			.stream()
			.forEach((functionName) -> {
				final SqlFunction function =
					FunctionCatalogUtils.toSqlFunction(
						catalog, functionName, catalog.getFunction(new ObjectPath(defaultDatabase, functionName)), typeFactory);
				sqlOperators.add(function);
			});
		return sqlOperators;
	}
}
