/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.fault.tolerant;

import org.apache.flink.metrics.Counter;
import org.apache.flink.metrics.MetricGroup;
import org.apache.flink.runtime.checkpoint.CheckpointOptions;
import org.apache.flink.runtime.jobgraph.JobVertexID;
import org.apache.flink.runtime.jobgraph.OperatorID;
import org.apache.flink.runtime.state.CheckpointStreamFactory;
import org.apache.flink.streaming.api.graph.StreamConfig;
import org.apache.flink.streaming.api.operators.ChainingStrategy;
import org.apache.flink.streaming.api.operators.OperatorSnapshotFutures;
import org.apache.flink.streaming.api.operators.Output;
import org.apache.flink.streaming.api.operators.StreamOperator;
import org.apache.flink.streaming.api.operators.Triggerable;
import org.apache.flink.streaming.runtime.streamrecord.StreamRecord;
import org.apache.flink.streaming.runtime.tasks.StreamTask;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A stream operator proxy which wraps a {@link StreamOperator}, catch all exception (except {@link NonnegligibleException})
 * of the underlying operator to prevent job failover.
 * Note: The proxy could not handle exception in the following case:
 * 1. FaultTolerantableAsyncWaitOperator, the exception is handled by {@link FaultTolerantableAsyncWaitOperator}.
 * 2. {@link Triggerable} operator, the exception happens in {@link Triggerable#onEventTime} and
 *    {@link Triggerable#onProcessingTime} is handled by {@link TriggerableOperator}.
 * 3. {@link ProcessingTimeCallbackOperator} operator, the exception happens in {@link ProcessingTimeCallbackOperator#onProcessingTime(long)}
 *    is handled by {@link ProcessingTimeCallbackOperator}.
 *
 * @param <OUT> The output type of the operator
 */
public abstract class AbstractStreamOperatorProxy<OUT> implements StreamOperator<OUT> {

	protected static final Logger LOG = LoggerFactory.getLogger(AbstractStreamOperatorProxy.class);

	protected transient Counter ignoreCnt;

	/** The name of underlying operator. */
	protected final String operatorName;

	/** Underlying operator. */
	protected final StreamOperator<OUT> operator;

	/** The task that contains this operator (and other operators in the same chain). */
	protected transient JobVertexID jobVertexId;

	public AbstractStreamOperatorProxy(String operatorName, StreamOperator<OUT> operator) {
		this.operatorName = operatorName;
		this.operator = operator;
	}

	@Override
	public void setup(StreamTask<?, ?> containingTask, StreamConfig config, Output<StreamRecord<OUT>> output) {
		jobVertexId = containingTask.getEnvironment().getJobVertexId();
		operator.setup(containingTask, config, new OutputProxy<>(output));
	}

	@Override
	public void open() throws Exception {
		operator.open();
		ignoreCnt = getMetricGroup().counter(FaultTolerantUtil.IGNORE_CNT_METRIC_NAME);
	}

	@Override
	public void close() throws Exception {
		try {
			operator.close();
		} catch (Exception e) {
			// Exclude runtime exception
			if (e instanceof NonnegligibleException) {
				throw e;
			} else {
				LOG.error("An exception is ignored when close operator [{}] of vertex [{}].\n" +
								"The output maybe discarded.",
						operatorName, jobVertexId, e);
				ignoreCnt.inc();
			}
		}
	}

	@Override
	public void dispose() throws Exception {
		operator.dispose();
	}

	@Override
	public void prepareSnapshotPreBarrier(long checkpointId) throws Exception {
		try {
			operator.prepareSnapshotPreBarrier(checkpointId);
		} catch (Exception e) {
			// Exclude runtime exception
			if (e instanceof NonnegligibleException) {
				throw e;
			} else {
				LOG.error("An exception is ignored when prepare snapshot preBarrier operator [{}] of vertex [{}].\n" +
								"The output maybe discarded.",
						operatorName, jobVertexId, e);
				ignoreCnt.inc();
			}
		}
	}

	@Override
	public OperatorSnapshotFutures snapshotState(long checkpointId, long timestamp, CheckpointOptions checkpointOptions,
			CheckpointStreamFactory storageLocation) throws Exception {
		return operator.snapshotState(checkpointId, timestamp, checkpointOptions, storageLocation);
	}

	@Override
	public void initializeState() throws Exception {
		operator.initializeState();
	}

	@Override
	public void setKeyContextElement1(StreamRecord<?> record) throws Exception {
		operator.setKeyContextElement1(record);
	}

	@Override
	public void setKeyContextElement2(StreamRecord<?> record) throws Exception {
		operator.setKeyContextElement2(record);
	}

	@Override
	public ChainingStrategy getChainingStrategy() {
		return operator.getChainingStrategy();
	}

	@Override
	public void setChainingStrategy(ChainingStrategy strategy) {
		operator.setChainingStrategy(strategy);
	}

	@Override
	public MetricGroup getMetricGroup() {
		return operator.getMetricGroup();
	}

	@Override
	public OperatorID getOperatorID() {
		return operator.getOperatorID();
	}

	@Override
	public boolean requireState() {
		return operator.requireState();
	}

	@Override
	public void notifyCheckpointComplete(long checkpointId) throws Exception {
		operator.notifyCheckpointComplete(checkpointId);
	}

	@Override
	public void setCurrentKey(Object key) {
		operator.setCurrentKey(key);
	}

	@Override
	public Object getCurrentKey() {
		return operator.getCurrentKey();
	}

}
