/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.fault.tolerant;

import org.apache.flink.streaming.api.operators.OneInputStreamOperator;
import org.apache.flink.streaming.api.operators.StreamOperator;
import org.apache.flink.streaming.api.operators.TwoInputStreamOperator;
import org.apache.flink.table.api.TableConfig;
import org.apache.flink.table.api.TableConfigOptions;
import org.apache.flink.table.runtime.SubstituteStreamOperator;

/**
 * Utility to support fault tolerant.
 */
public class FaultTolerantUtil {

	/** Name of metric which to count ignore exception times. */
	public static final String IGNORE_CNT_METRIC_NAME = "ignore_cnt";

	/**
	 * Add fault tolerant proxy on input {@link OneInputStreamOperator} if needed.
	 *
	 * @param operator underlying operator
	 * @param operatorName name of underlying operator
	 * @param tableConfig table configuration
	 * @param <IN> The input type of the operator
	 * @param <OUT> The output type of the operator
	 * @return proxy operator which wraps the input operator if there is need to support fault tolerant, else return
	 * input operator.
	 */
	public static <IN, OUT> OneInputStreamOperator<IN, OUT> addFaultTolerantProxyIfNeed(
			OneInputStreamOperator<IN, OUT> operator,
			String operatorName,
			TableConfig tableConfig) {
		return addProxyIfNeed(operator, operatorName, tableConfig);
	}

	/**
	 * Add fault tolerant proxy on input {@link TwoInputStreamOperator} if needed.
	 *
	 * @param operator underlying operator
	 * @param operatorName name of underlying operator
	 * @param tableConfig table configuration
	 * @param <IN1> The input type of the operator
	 * @param <IN2> The input type of the operator
	 * @param <OUT> The output type of the operator
	 * @return proxy operator which wraps the input operator if there is need to support fault tolerant, else return
	 * input operator.
	 */
	public static <IN1, IN2, OUT> TwoInputStreamOperator<IN1, IN2, OUT> addFaultTolerantProxyIfNeed(
			TwoInputStreamOperator<IN1, IN2, OUT> operator,
			String operatorName,
			TableConfig tableConfig) {
		return addProxyIfNeed(operator, operatorName, tableConfig);
	}

	private static <OUT, T extends StreamOperator<OUT>> T addProxyIfNeed(T operator, String operatorName,
			TableConfig tableConfig) {
		boolean faultTolerantEnable = tableConfig.getConf()
				.getBoolean(TableConfigOptions.SQL_EXEC_FAULT_TOLERANCE_ENABLED);
		if (!faultTolerantEnable) {
			return operator;
		}
		if (operator instanceof FaultTolerantableOperator) {
			((FaultTolerantableOperator) operator).enableFaultTolerant();
		}
		if (operator instanceof SubstituteStreamOperator || operator instanceof FaultTolerantableAsyncWaitOperator) {
			return operator;
		}
		if (operator instanceof OneInputStreamOperator) {
			return (T) new OneInputOperatorProxy(operatorName, (OneInputStreamOperator) operator);
		} else {
			return (T) new TwoInputOperatorProxy(operatorName, (TwoInputStreamOperator) operator);
		}
	}

	private FaultTolerantUtil() {
		// Deprecates fault constructor
	}
}
