/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.fault.tolerant;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.metrics.Counter;
import org.apache.flink.streaming.api.functions.co.CoProcessFunction;
import org.apache.flink.streaming.api.operators.InternalTimerService;
import org.apache.flink.streaming.api.operators.Triggerable;

/**
 * An subclass of {@link org.apache.flink.streaming.api.operators.co.KeyedCoProcessOperator} which is fault tolerant.
 * Note: the class does not inherit {@link TriggerableOperator} because a class cannot inherit two super class in JAVA.
 */
public class FaultTolerantableKeyedCoProcessOperator<K, IN1, IN2, OUT>
		extends org.apache.flink.streaming.api.operators.co.KeyedCoProcessOperator<K, IN1, IN2, OUT>
		implements FaultTolerantableOperator<OUT> {

	private static final long serialVersionUID = 1L;

	private boolean faultTolerantEnable = false;

	public FaultTolerantableKeyedCoProcessOperator(CoProcessFunction<IN1, IN2, OUT> flatMapper) {
		super(flatMapper);
	}

	@Override
	public void enableFaultTolerant() {
		this.faultTolerantEnable = true;
	}

	@Override
	public <K, N> InternalTimerService<N> getInternalTimerService(String name, TypeSerializer<N> namespaceSerializer,
			Triggerable<K, N> triggerable) {
		Triggerable<K, N> t;
		if (faultTolerantEnable) {
			Counter ignoreCnt = getMetricGroup().counter(FaultTolerantUtil.IGNORE_CNT_METRIC_NAME);
			t = new TriggerableOperator.TriggerableProxy(ignoreCnt, triggerable);
		} else {
			t = triggerable;
		}
		return super.getInternalTimerService(name, namespaceSerializer, t);
	}
}
