/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.fault.tolerant;

import org.apache.flink.annotation.Internal;
import org.apache.flink.streaming.api.operators.Output;
import org.apache.flink.streaming.api.watermark.Watermark;
import org.apache.flink.streaming.runtime.streamrecord.LatencyMarker;
import org.apache.flink.streaming.runtime.streamrecord.StreamRecord;
import org.apache.flink.util.OutputTag;

/**
 * An output proxy which wraps the real output. The proxy catches all exception of the underlying output when emits a
 * record, then wraps the exception in a {@link NonnegligibleException}.
 *
 * @param <T> The type of the elements that can be emitted.
 */
@Internal
public class OutputProxy<T> extends CollectorProxy<T> implements Output<T> {

	public OutputProxy(Output<T> output) {
		super(output);
	}

	@Override
	public <X> void collect(OutputTag<X> outputTag, StreamRecord<X> record) {
		try {
			((Output) collector).collect(outputTag, record);
		} catch (Throwable e) {
			throw new NonnegligibleException(e);
		}
	}

	@Override
	public void emitWatermark(Watermark mark) {
		((Output) collector).emitWatermark(mark);
	}

	@Override
	public void emitLatencyMarker(LatencyMarker latencyMarker) {
		((Output) collector).emitLatencyMarker(latencyMarker);
	}

}
