/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.functions.utils;

import org.apache.flink.table.dataformat.BinaryString;
import org.apache.flink.table.dataformat.Decimal;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;

/**
 * Murmur3 is successor to Murmur2 fast non-crytographic hash algorithms.
 *
 * <p>Murmur3 32 and 128 bit variants.
 * 32-bit Java port of https://code.google.com/p/smhasher/source/browse/trunk/MurmurHash3.cpp#94
 * 128-bit Java port of https://code.google.com/p/smhasher/source/browse/trunk/MurmurHash3.cpp#255
 *
 * <p>This is a public domain code with no copyrights.
 * From homepage of MurmurHash (https://code.google.com/p/smhasher/),
 * "All MurmurHash versions are public domain software, and the author disclaims all copyright
 * to their code."
 *
 * <p>This is referred from Hive: org.apache.hive.common.util.Murmur3.
 */
public class Murmur3Utils {
	// Constants for 32 bit variant
	private static final int C1_32 = 0xcc9e2d51;
	private static final int C2_32 = 0x1b873593;
	private static final int R1_32 = 15;
	private static final int R2_32 = 13;
	private static final int M_32 = 5;
	private static final int N_32 = 0xe6546b64;

	// Constants for 128 bit variant
	private static final long C1 = 0x87c37b91114253d5L;
	private static final long C2 = 0x4cf5ad432745937fL;
	private static final int R1 = 31;
	private static final int R2 = 27;
	private static final int M = 5;
	private static final int N1 = 0x52dce729;

	public static final int DEFAULT_SEED = 104729;

	public static int hash32(Object l0) {
		if (l0 instanceof Boolean) {
			long val = 0;
			if ((boolean) l0) {
				val = 1;
			}
			return hash32(val);
		} else if (l0 instanceof Float) {
			return hash32(Float.floatToIntBits((Float) l0));
		} else if (l0 instanceof Double) {
			return hash32(Double.doubleToLongBits((Double) l0));
		} else if (l0 instanceof Decimal) {
			return hash32(((Decimal) l0).toUnscaledBytes());
		} else if (l0 instanceof Date || l0 instanceof Time || l0 instanceof Timestamp) {
			return hash32(((Date) l0).getTime());
		} else if (l0 instanceof BinaryString) {
			return hash32(((BinaryString) l0).getBytes());
		} else {
			throw new IllegalArgumentException(
				"MURMUR_HASH: Unsupported operand type: " + l0.getClass());
		}
	}

	public static long hash64(Object l0) {
		if (l0 instanceof Boolean) {
			short val = 0;
			if ((boolean) l0) {
				val = 1;
			}
			return hash64(val);
		} else if (l0 instanceof Float) {
			return hash64(Float.floatToIntBits((Float) l0));
		} else if (l0 instanceof Double) {
			return hash64(Double.doubleToLongBits((Double) l0));
		} else if (l0 instanceof Decimal) {
			return hash64(((Decimal) l0).toUnscaledBytes());
		} else if (l0 instanceof Date || l0 instanceof Time || l0 instanceof Timestamp) {
			return hash64(((Date) l0).getTime());
		} else if (l0 instanceof BinaryString) {
			return hash64(((BinaryString) l0).getBytes());
		} else {
			throw new IllegalArgumentException(
				"MURMUR_HASH: Unsupported operand type: " + l0.getClass());
		}
	}

	public static int hash32(long l0) {
		return hash32(l0, DEFAULT_SEED);
	}

	/**
	 * Murmur3 32-bit variant.
	 */
	public static int hash32(long l0, int seed) {
		int hash = seed;
		final long r0 = Long.reverseBytes(l0);

		hash = mix32((int) r0, hash);
		hash = mix32((int) (r0 >>> 32), hash);

		return fmix32(Long.BYTES, hash);
	}

	/**
	 * Murmur3 32-bit variant.
	 *
	 * @param data - input byte array
	 * @return - hashcode
	 */
	public static int hash32(byte[] data) {
		return hash32(data, 0, data.length, DEFAULT_SEED);
	}

	/**
	 * Murmur3 32-bit variant.
	 *
	 * @param data - input byte array
	 * @param length - length of array
	 * @return - hashcode
	 */
	public static int hash32(byte[] data, int length) {
		return hash32(data, 0, length, DEFAULT_SEED);
	}

	/**
	 * Murmur3 32-bit variant.
	 *
	 * @param data   - input byte array
	 * @param length - length of array
	 * @param seed   - seed. (default 0)
	 * @return - hashcode
	 */
	public static int hash32(byte[] data, int length, int seed) {
		return hash32(data, 0, length, seed);
	}

	/**
	 * Murmur3 32-bit variant.
	 *
	 * @param data   - input byte array
	 * @param offset - offset of data
	 * @param length - length of array
	 * @param seed   - seed. (default 0)
	 * @return - hashcode
	 */
	public static int hash32(byte[] data, int offset, int length, int seed) {
		int hash = seed;
		final int nblocks = length >> 2;

		// body
		for (int i = 0; i < nblocks; i++) {
			int i4 = i << 2;
			int k = (data[offset + i4] & 0xff)
				| ((data[offset + i4 + 1] & 0xff) << 8)
				| ((data[offset + i4 + 2] & 0xff) << 16)
				| ((data[offset + i4 + 3] & 0xff) << 24);

			hash = mix32(k, hash);
		}

		// tail
		int idx = nblocks << 2;
		int k1 = 0;
		int tailLen = length - idx;
		if (tailLen > 2) {
			k1 ^= data[offset + idx + 2] << 16;
		}
		if (tailLen > 1) {
			k1 ^= data[offset + idx + 1] << 8;
		}
		if (tailLen > 0) {
			k1 ^= data[offset + idx];
			// mix functions
			k1 *= C1_32;
			k1 = Integer.rotateLeft(k1, R1_32);
			k1 *= C2_32;
			hash ^= k1;
		}

		return fmix32(length, hash);
	}

	private static int mix32(int k, int hash) {
		k *= C1_32;
		k = Integer.rotateLeft(k, R1_32);
		k *= C2_32;
		hash ^= k;
		return Integer.rotateLeft(hash, R2_32) * M_32 + N_32;
	}

	private static int fmix32(int length, int hash) {
		hash ^= length;
		hash ^= (hash >>> 16);
		hash *= 0x85ebca6b;
		hash ^= (hash >>> 13);
		hash *= 0xc2b2ae35;
		hash ^= (hash >>> 16);

		return hash;
	}

	/**
	 * Murmur3 64-bit variant. This is essentially MSB 8 bytes of Murmur3 128-bit variant.
	 *
	 * @param data - input byte array
	 * @return - hashcode
	 */
	public static long hash64(byte[] data) {
		return hash64(data, 0, data.length, DEFAULT_SEED);
	}

	public static long hash64(long data) {
		long hash = DEFAULT_SEED;
		long k = Long.reverseBytes(data);
		int length = Long.BYTES;
		// mix functions
		k *= C1;
		k = Long.rotateLeft(k, R1);
		k *= C2;
		hash ^= k;
		hash = Long.rotateLeft(hash, R2) * M + N1;
		// finalization
		hash ^= length;
		hash = fmix64(hash);
		return hash;
	}

	public static long hash64(int data) {
		long k1 = Integer.reverseBytes(data) & (-1L >>> 32);
		int length = Integer.BYTES;
		long hash = DEFAULT_SEED;
		k1 *= C1;
		k1 = Long.rotateLeft(k1, R1);
		k1 *= C2;
		hash ^= k1;
		// finalization
		hash ^= length;
		hash = fmix64(hash);
		return hash;
	}

	public static long hash64(short data) {
		long hash = DEFAULT_SEED;
		long k1 = 0;
		k1 ^= ((long) data & 0xff) << 8;
		k1 ^= ((long) ((data & 0xFF00) >> 8) & 0xff);
		k1 *= C1;
		k1 = Long.rotateLeft(k1, R1);
		k1 *= C2;
		hash ^= k1;

		// finalization
		hash ^= Short.BYTES;
		hash = fmix64(hash);
		return hash;
	}

	public static long hash64(byte[] data, int offset, int length) {
		return hash64(data, offset, length, DEFAULT_SEED);
	}

	/**
	 * Murmur3 64-bit variant. This is essentially MSB 8 bytes of Murmur3 128-bit variant.
	 *
	 * @param data   - input byte array
	 * @param length - length of array
	 * @param seed   - seed. (default is 0)
	 * @return - hashcode
	 */
	public static long hash64(byte[] data, int offset, int length, int seed) {
		long hash = seed;
		final int nblocks = length >> 3;

		// body
		for (int i = 0; i < nblocks; i++) {
			final int i8 = i << 3;
			long k = ((long) data[offset + i8] & 0xff)
				| (((long) data[offset + i8 + 1] & 0xff) << 8)
				| (((long) data[offset + i8 + 2] & 0xff) << 16)
				| (((long) data[offset + i8 + 3] & 0xff) << 24)
				| (((long) data[offset + i8 + 4] & 0xff) << 32)
				| (((long) data[offset + i8 + 5] & 0xff) << 40)
				| (((long) data[offset + i8 + 6] & 0xff) << 48)
				| (((long) data[offset + i8 + 7] & 0xff) << 56);

			// mix functions
			k *= C1;
			k = Long.rotateLeft(k, R1);
			k *= C2;
			hash ^= k;
			hash = Long.rotateLeft(hash, R2) * M + N1;
		}

		// tail
		long k1 = 0;
		int tailStart = nblocks << 3;
		int tailLen = length - tailStart;
		if (tailLen > 6) {
			k1 ^= ((long) data[offset + tailStart + 6] & 0xff) << 48;
		}
		if (tailLen > 5) {
			k1 ^= ((long) data[offset + tailStart + 5] & 0xff) << 40;
		}
		if (tailLen > 4) {
			k1 ^= ((long) data[offset + tailStart + 4] & 0xff) << 32;
		}
		if (tailLen > 3) {
			k1 ^= ((long) data[offset + tailStart + 3] & 0xff) << 24;
		}
		if (tailLen > 2) {
			k1 ^= ((long) data[offset + tailStart + 2] & 0xff) << 16;
		}
		if (tailLen > 1) {
			k1 ^= ((long) data[offset + tailStart + 1] & 0xff) << 8;
		}
		if (tailLen > 0) {
			k1 ^= ((long) data[offset + tailStart] & 0xff);
			k1 *= C1;
			k1 = Long.rotateLeft(k1, R1);
			k1 *= C2;
			hash ^= k1;
		}

		// finalization
		hash ^= length;
		hash = fmix64(hash);

		return hash;
	}

	private static long fmix64(long h) {
		h ^= (h >>> 33);
		h *= 0xff51afd7ed558ccdL;
		h ^= (h >>> 33);
		h *= 0xc4ceb9fe1a85ec53L;
		h ^= (h >>> 33);
		return h;
	}
}
