/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.	See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.	You may obtain a copy of the License at
 *
 *		http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.util;

import org.apache.flink.core.memory.MemorySegment;
import org.apache.flink.runtime.io.disk.iomanager.IOManager;
import org.apache.flink.runtime.memory.MemoryManager;
import org.apache.flink.table.dataformat.BaseRow;
import org.apache.flink.table.typeutils.AbstractRowSerializer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

/**
 * Java list resettable buffer.
 */
public class ResettableListBuffer implements ResettableRowBuffer<BaseRow> {

	private final MemoryManager memoryManager;
	private final IOManager ioManager;
	private final List<MemorySegment> memory;
	private final AbstractRowSerializer<BaseRow> serializer;
	private final int maxSize;

	private boolean fallback;
	private ArrayList<BaseRow> list;
	private ResettableExternalBuffer externalBuffer;

	public ResettableListBuffer(
			MemoryManager memoryManager,
			IOManager ioManager,
			List<MemorySegment> memory,
			AbstractRowSerializer serializer) {
		this(memoryManager, ioManager, memory, serializer, 2048);
	}

	public ResettableListBuffer(
			MemoryManager memoryManager,
			IOManager ioManager,
			List<MemorySegment> memory,
			AbstractRowSerializer serializer,
			int maxSize) {
		this.memoryManager = memoryManager;
		this.ioManager = ioManager;
		this.memory = memory;
		this.serializer = serializer;
		this.maxSize = maxSize;
		this.list = new ArrayList<>(maxSize);
		this.fallback = false;
	}

	private void fallbackToExternal() throws IOException {
		this.externalBuffer = new ResettableExternalBuffer(memoryManager, ioManager, memory, serializer);
		for (BaseRow row : list) {
			externalBuffer.add(row);
		}
		this.list = null;
		this.fallback = true;
	}

	@Override
	public void reset() {
		if (fallback) {
			this.externalBuffer.reset();
		} else {
			list.clear();
		}
	}

	@Override
	public void add(BaseRow row) throws IOException {
		if (fallback) {
			this.externalBuffer.add(row);
		} else {
			list.add(serializer.copy(row));

			if (list.size() > maxSize) {
				fallbackToExternal();
			}
		}
	}

	@Override
	public int size() {
		return fallback ? externalBuffer.size() : list.size();
	}

	@Override
	public BufferIterator newIterator() {
		BufferIterator iter = new BufferIterator();
		if (fallback) {
			iter.externalIter = externalBuffer.newIterator();
		} else {
			iter.memIter = list.iterator();
		}
		return iter;
	}

	@Override
	public BufferIterator newIterator(int beginRow) {
		throw new UnsupportedOperationException();
	}

	@Override
	public void close() {
		if (fallback) {
			this.externalBuffer.close();
		} else {
			list = null;
			memoryManager.release(memory);
		}
	}

	/**
	 * Iterator.
	 */
	public class BufferIterator implements ResettableIterator<BaseRow> {

		private Iterator<BaseRow> memIter;
		private BaseRow currRow;
		private ResettableExternalBuffer.BufferIterator externalIter;

		@Override
		public void reset() throws IOException {
			if (fallback) {
				externalIter.reset();
			} else {
				memIter = list.iterator();
			}
		}

		@Override
		public void close() throws IOException {
			if (fallback) {
				externalIter.close();
			}
		}

		@Override
		public boolean advanceNext() {
			if (fallback) {
				return externalIter.advanceNext();
			} else {
				boolean has = memIter.hasNext();
				if (has) {
					currRow = memIter.next();
				}
				return has;
			}
		}

		@Override
		public BaseRow getRow() {
			return fallback ? externalIter.getRow() : currRow;
		}
	}
}
