/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.runtime.window;

import org.apache.flink.api.common.typeutils.TypeSerializer;
import org.apache.flink.metrics.Gauge;
import org.apache.flink.runtime.state.OperatorStateBackend;
import org.apache.flink.streaming.api.operators.AbstractStreamOperator;
import org.apache.flink.streaming.api.operators.ChainingStrategy;
import org.apache.flink.streaming.api.operators.OneInputStreamOperator;
import org.apache.flink.streaming.api.operators.TimestampedCollector;
import org.apache.flink.streaming.api.watermark.Watermark;
import org.apache.flink.table.api.window.TimeWindow;
import org.apache.flink.table.dataformat.BaseRow;
import org.apache.flink.table.runtime.functions.ExecutionContext;
import org.apache.flink.table.runtime.functions.ExecutionContextImpl;
import org.apache.flink.table.runtime.window.aligned.AlignedWindowAggregator;
import org.apache.flink.table.runtime.window.aligned.AlignedWindowTrigger;
import org.apache.flink.table.runtime.window.assigners.SlidingWindowAssigner;
import org.apache.flink.table.runtime.window.assigners.TumblingWindowAssigner;
import org.apache.flink.table.runtime.window.assigners.WindowAssigner;
import org.apache.flink.util.Collector;

import java.util.Collection;

import static org.apache.flink.util.Preconditions.checkArgument;

/**
 * An operator that implements the logic for windowing based on a {@link AlignedWindowAggregator} and
 * {@link WindowAssigner} and {@link AlignedWindowTrigger}.
 *
 * <p>This window operator is an optimized operator for aligned windows. For aligned windows we can
 * make sure what's the next triggered window, what's the next triggered watermark. So that we can
 * avoid to rely on registering a lot of timer triggers to trigger window and get the
 * higher performance.
 *
 * <p>The {@link AlignedWindowAggregator} is used to access & maintain window states and fire window
 * results. It can be a buffered or memory-managed-buffered implementation to reduce state access.
 */
public abstract class AbstractAlignedWindowOperator
	extends AbstractStreamOperator<BaseRow>
	implements OneInputStreamOperator<BaseRow, BaseRow> {

	private static final long serialVersionUID = 1L;

	protected static final String LATE_ELEMENTS_DROPPED_METRIC_NAME = "numLateRecordsDropped";
	protected static final String LATE_ELEMENTS_DROPPED_RATE_METRIC_NAME = "lateRecordsDroppedRate";
	protected static final String WATERMARK_LATENCY_METRIC_NAME = "watermarkLatency";

	// ------------------------------------------------------------------------
	// Configuration values and user functions
	// ------------------------------------------------------------------------

	protected final AlignedWindowAggregator<BaseRow, TimeWindow, BaseRow> windowRunner;

	protected final WindowAssigner<TimeWindow> windowAssigner;

	protected final AlignedWindowTrigger windowTrigger;

	protected final int rowtimeIndex;

	// --------------------------------------------------------------------------------

	protected transient long nextTriggerTime;

	protected transient TimeWindow nextTriggerWindow;

	/**
	 * This is given to the {@code InternalWindowFunction} for emitting elements with a given
	 * timestamp.
	 */
	protected transient TimestampedCollector<BaseRow> collector;

	/** Flag to prevent duplicate function.close() calls in close() and dispose(). */
	private transient boolean functionsClosed = false;

	private transient Gauge<Long> watermarkLatency;

	private transient long currentWatermark;

	public AbstractAlignedWindowOperator(
		AlignedWindowAggregator<BaseRow, TimeWindow, BaseRow> windowRunner,
		WindowAssigner<TimeWindow> windowAssigner,
		AlignedWindowTrigger windowTrigger,
		int rowtimeIndex) {
		this.windowRunner = windowRunner;
		this.windowTrigger = windowTrigger;

		// rowtime index should >= 0 when in event time mode
		if (!(windowAssigner instanceof SlidingWindowAssigner) && !(windowAssigner instanceof TumblingWindowAssigner)) {
			throw new IllegalArgumentException("Currently aligned window only support sliding and tumbling windows.");
		}
		checkArgument(!windowAssigner.isEventTime() || rowtimeIndex >= 0);
		this.windowAssigner = windowAssigner;
		this.rowtimeIndex = rowtimeIndex;

		setChainingStrategy(ChainingStrategy.ALWAYS);
	}

	@Override
	public void open() throws Exception {
		super.open();

		this.collector = new TimestampedCollector<>(output);
		this.collector.eraseTimestamp();

		this.nextTriggerTime = Long.MIN_VALUE;
		this.nextTriggerWindow = null;

		TypeSerializer<TimeWindow> windowSerializer = new TimeWindow.Serializer();
		ExecutionContext ctx = new ExecutionContextImpl(this, getRuntimeContext(), windowSerializer);
		windowRunner.open(new WindowContextImpl(ctx, collector));

		// metrics
		this.watermarkLatency = metrics.gauge(WATERMARK_LATENCY_METRIC_NAME, () -> {
			long watermark = this.currentWatermark;
			if (watermark < 0) {
				return 0L;
			} else {
				return System.currentTimeMillis() - watermark;
			}
		});
	}

	@Override
	public void close() throws Exception {
		super.close();
		collector = null;
		functionsClosed = true;
		windowRunner.close();
	}

	@Override
	public void dispose() throws Exception {
		super.dispose();
		collector = null;
		if (!functionsClosed) {
			functionsClosed = true;
			windowRunner.close();
		}
	}

	@Override
	public void processWatermark(Watermark mark) throws Exception {
		this.currentWatermark = mark.getTimestamp();
		advanceWatermark(currentWatermark);
		super.processWatermark(mark);
	}

	protected abstract void advanceWatermark(long watermark) throws Exception;

	protected abstract BaseRow getKey(final BaseRow input) throws Exception;

	/**
	 * Whether a window is late (can be expired).
	 *
	 * <p>NOTE: Currently, the implementation of this method is totally the same as
	 * {@link #needTriggerWindow(TimeWindow, long)}. But will be different in the
	 * future when take allowlateness into account.
	 */
	protected boolean isWindowLate(TimeWindow window, long watermark) {
		long cleanupTime = window.maxTimestamp();
		return cleanupTime <= watermark;
	}

	/**
	 * Whether a window can be triggered, i.e. watermark excesses window's max timestamp.
	 */
	protected boolean needTriggerWindow(TimeWindow window, long watermark) {
		long cleanupTime = window.maxTimestamp();
		return cleanupTime <= watermark;
	}

	protected void expireWindows(Collection<TimeWindow> windowsToExpire) throws Exception {
		for (TimeWindow window : windowsToExpire) {
			windowRunner.expireWindow(window);
		}
	}

	public long getCurrentWatermark() {
		return this.currentWatermark;
	}

	@Override
	public void endInput() throws Exception {
		// nothing to do
	}

	private class WindowContextImpl implements AlignedWindowAggregator.Context {

		private final ExecutionContext executionContext;
		private final Collector<BaseRow> out;

		public WindowContextImpl(ExecutionContext executionContext, Collector<BaseRow> out) {
			this.executionContext = executionContext;
			this.out = out;
		}

		@Override
		public ExecutionContext getExecutionContext() throws Exception {
			return this.executionContext;
		}

		@Override
		public Collector<BaseRow> getCollector() throws Exception {
			return this.out;
		}

		@Override
		public OperatorStateBackend getOpStateStore() throws Exception {
			return getOperatorStateBackend();
		}
	}

	// ------------------------------------------------------------------------------
	// Visible For Testing
	// ------------------------------------------------------------------------------

	protected Gauge<Long> getWatermarkLatency() {
		return watermarkLatency;
	}
}
